/*
  bbdriver.c
  Jan 20, 2002

  for K2915,CC77,VMEMM
*/

#define ADDRESS (MEM*0x100000)
#define CADDRESS (MEM*0x100000 | 0x8000)
#define DMAADDRESS (MEM*0x100000 | 0x40000)

#include <string.h>

#ifdef K2915
#include "k2915.c"
#endif

#ifdef CC7700
#include "cc7700.c"
#endif

#ifdef VMEMM
#include "vmemm.c"
#include "v785.h"
#endif

void mem_clear(void);
void init_driver(int irq,int ior,int pbor);
void end_driver(void);
void init_block(void);
void init_event(short fid);
void init_segment(short segid);
int end_block(void);
int end_event(void);
int end_segment(void);
int read_segdata(short c,short n,short a,short f);
int read_segndata(short len,short c,short n,short a,short f);
int read_segsdata(short len,short c,short n,short a,short f);
int read_segbdata(short len,short c,short n,short a,short f);
int read_segmod(short len,short segid,short c,short n,short a,short f);
int read_segsmod(short len,short segid,short c,short n,short a,short f);
int read_segbmod(short len,short segid,short c,short n,short a,short f);
int read_dsegbmod(short len,short segid,short c,short n,short a,short f);
int read_segmemmod(short segid,short c,short n);
int read_dsegmemmod(short segid,short c,short n);
int read_scaler(short pos,short len,short c,short n);
int read_sscaler(short pos,short len,short c,short n);
int read_pedestal(short type,short c,short n,short vsn);
void write_mod(short c,short n,short a,short f,short *data);
void control_mod(short c,short n,short a,short f);
void init_shmem(void);
void dinit_shmem(int sh);
void change_shmem(int sh);
void dclear_shmem(int sh);
void clear_shmem(void);
void put_data(char flag);
void put_smem(void);
int get_mp(void);
void print_shmem(int lines);

/* VME Series */
#ifdef VMEMM
int v785_segdata(unsigned long maddr);
int v785_segmod(short segid,unsigned long maddr);
#endif

static unsigned short *data,*dmadata,*cdata;
static int mp,eventid,eventhmp,eventsize,segmenthmp,segmentsize;
int i;


void mem_clear(void){
  data = __va(ADDRESS);
  memset(data,0,0x4004);                   /* Memory set to '0' */
}

void init_driver(int irq,int ior,int pbor){
  data = __va(ADDRESS);
  cdata = __va(CADDRESS);
  dmadata = __va(DMAADDRESS);
  init_register(irq,ior,pbor);
  memset(data,0,0x8008);                   /* Memory set to '0' */
}

void end_driver(void){
  cdata[0] = 0xeeee;
}

void init_block(void){
  memset(data,0,4);                       /* Memory set to '0' */
  mp = 4;
  eventid = 0;
}

void init_pblock(void){
  data[0] = 0x0010;
  memset(data+1,0,3);                       /* Memory set to '0' */
  mp = 4;
  eventid = 0;
}

void init_event(short fid){
  eventid++;                               /* Count up event ID */
  eventhmp = mp;                           /* Store mp for event size */
  eventsize = 3;                           /* Initialize event size */
  mp++;
  data[mp++] = fid;                        /* Write FID */
  data[mp++] = eventid;                    /* Write event ID */
}

void init_segment(short segid){
  segmenthmp = mp;                         /* Store mp for segment size */
  segmentsize = 2;                         /* Initialize segment size */
  mp++;
  data[mp++] = segid;                      /* Write segment ID */
}

int end_block(void){
  data[mp++] = 0xffff;                     /* Write END */
  data[mp++] = 0xffff;                     /* Write END */

  return mp;
}

int end_event(void){
  data[eventhmp] = eventsize | 0x8000;     /* Write event size */
  return mp;
}

int end_segment(void){
  data[segmenthmp] = segmentsize;          /* Write segment size */
  eventsize += segmentsize;                /* Count up event size */

  return mp;
}

int read_segdata(short c,short n,short a,short f){
  read16(CNAFGEN(c,n,a,f),data+mp);        /* Read data */
  mp++;
  segmentsize++;                           /* Count up segment size */

  return segmentsize;
}

int read_segndata(short len,short c,short n,short a,short f){
  for(i=0;i<len;i++){
    read16(CNAFGEN(c,n,a,f),data+mp);        /* Read data */
    a++;
    mp++;
    segmentsize++;                           /* Count up segment size */
  }

  return segmentsize;
}

int read_segsdata(short len,short c,short n,short a,short f){
#ifdef K2915
  block_read16(QSCAN,CNAFGEN(c,n,a,f),data+mp,len); /* Read data */
  mp += len;
  segmentsize += len;
#else
  for(i=0;i<len;i++){
    read16(CNAFGEN(c,n,a,f),data+mp);        /* Read data */
    a++;
    mp++;
    segmentsize++;                           /* Count up segment size */
  }
#endif

  return segmentsize;
}

int read_segbdata(short len,short c,short n,short a,short f){
  block_read16(QIGNORE,CNAFGEN(c,n,a,f),data+mp,len); /* Read data */
  mp += len;
  segmentsize += len;

  return segmentsize;
}

int read_segmod(short len,short segid,short c,short n,short a,short f){
  data[mp++] = len+2;                      /* Write word count */
  data[mp++] = segid;                      /* Write segment ID */
  for(i=0;i<len;i++){
    read16(CNAFGEN(c,n,a,f),data+mp);      /* Read data */
    mp++;
    a++;
  }

  eventsize += len + 2;                    /* Count up event size */

  return len + 2;
}

int read_segsmod(short len,short segid,short c,short n,short a,short f){
  segmenthmp = mp;
  mp++;
  data[mp++] = segid;                      /* Write segment ID */
  block_read16(QSCAN,CNAFGEN(c,n,a,f),data+mp,len); /* Read data */
  mp += len;
  data[segmenthmp] = len + 2;              /* Write word count */
  eventsize += len + 2;                    /* Count up event size */

  return len + 2;
}

int read_segbmod(short len,short segid,short c,short n,short a,short f){
  segmenthmp = mp;
  mp++;
  data[mp++] = segid;                      /* Write segment ID */
  block_read16(QIGNORE,CNAFGEN(c,n,a,f),data+mp,len); /* Read data */
  mp += len;
  data[segmenthmp] = len + 2;              /* Write word count */
  eventsize += len + 2;                    /* Count up event size */

  return len + 2;
}

int read_dsegbmod(short len,short segid,short c,short n,short a,short f){
  segmenthmp = mp;
  mp++;
  data[mp++] = segid;                      /* Write segment ID */
  dma_block_read16(QSTOP,CNAFGEN(c,n,a,f),len); /* Read data */
  memcpy(data+mp,dmadata,len*2);           /* Data copy */
  mp += len;
  data[segmenthmp] = len + 2;              /* Write word count */
  eventsize += len + 2;                    /* Count up event size */

  return len + 2;
}

int read_segmemmod(short segid,short c,short n){
  short wdata,wordcnt;

  wdata = 0x0001;
  write16(CNAFGEN(c,n,1,17),&wdata);       /* Change to CAMAC-mode */
  read16(CNAFGEN(c,n,0,1),&wordcnt);       /* Read word count */
  wdata = 0x0000;
  write16(CNAFGEN(c,n,0,17),&wdata);       /* Reset read buffer pointer */
  data[mp++] = wordcnt+3;                  /* Write word count */
  data[mp++] = segid;                      /* Write segment ID */
  data[mp++] = wordcnt;                    /* Write FERA word count */
  if(wordcnt != 0){
    block_read16(QIGNORE,CNAFGEN(c,n,0,0),data+mp,wordcnt); /* Read */
  }
  mp += wordcnt;
  wdata = 0x0003;
  write16(CNAFGEN(c,n,1,17),&wdata);       /* Change to ECL-mode */

  eventsize += wordcnt + 3;

  return wordcnt + 3;
}

int read_dsegmemmod(short segid,short c,short n){
  short wdata,wordcnt;

  wdata = 0x0001;
  write16(CNAFGEN(c,n,1,17),&wdata);       /* Change to CAMAC-mode */
  read16(CNAFGEN(c,n,0,1),&wordcnt);       /* Read word count */
  wdata = 0x0000;
  write16(CNAFGEN(c,n,0,17),&wdata);       /* Reset read buffer pointer */
  data[mp++] = wordcnt+3;                  /* Write word count */
  data[mp++] = segid;                      /* Write segment ID */
  data[mp++] = wordcnt;                    /* Write FERA word count */
  if(wordcnt != 0){
    dma_block_read16(QSTOP,CNAFGEN(c,n,0,0),wordcnt); /* Read */
    memcpy(data+mp,__va(DMAADDRESS),wordcnt*2);     /* Data copy */
  }
  mp += wordcnt;
  wdata = 0x0003;
  write16(CNAFGEN(c,n,1,17),&wdata);       /* Change to ECL-mode */

  eventsize += wordcnt + 3;

  return wordcnt + 3;
}

int read_scaler(short pos,short len,short c,short n){
  mp = 0x2000 - (len * 2 * pos);           /* Set mp */
  for(i=0;i<len;i++){
    read24(CNAFGEN(c,n,i,0),data+mp);      /* Read Scaler data */
    mp += 2;
  }

  return len * 2;
}

int read_sscaler(short pos,short len,short c,short n){
  mp = 0x2000 - (len * 2 * pos);           /* Set mp */

  block_read24(QSCAN,CNAFGEN(c,n,0,0),(unsigned int*)(data+mp),len);
  /* Read data 24Bit Block Transfer Mode*/

  return len * 2;
}

int read_pedestal(short type,short c,short n,short vsn){
  short len;
  /* type 0 : 4300B
          1 : 3351
  */
  data[mp++] = type;
  data[mp++] = (c<<8)|n;
  data[mp++] = vsn;
  if(type == 0){
    len = 16;
  }else{
    len = 8;
  }
  block_read16(QIGNORE,CNAFGEN(c,n,0,2),data+mp,len); /* Read */
  mp += len;

  return mp;
}

void write_mod(short c,short n,short a,short f,short *data){
  write16(CNAFGEN(c,n,a,f),data);
}

void control_mod(short c,short n,short a,short f){
  control(CNAFGEN(c,n,a,f));
}

void init_shmem(void){
  char put = 0x01;
  cdata[0] = 0x0101;
  rtf_put(3,&put,1);
}

void dinit_shmem(int sh){
  if(sh == 0){
    cdata[0] = 0x0101;
  }else{
    cdata[1] = 0x0202;
  }
}

void change_shmem(int sh){
  if(sh == 0 ){
    data = __va(ADDRESS);
    /* memset(data,0,0x4000); */
  }else{
    data = __va((ADDRESS | 0x4000));
    /* memset(data,0,0x4000); */
  }
}

void clear_shmem(void){
  cdata[0] = 0x0000;
}

void dclear_shmem(int sh){
  if(sh == 0){
    cdata[0] = 0x0000;
  }else{
    cdata[1] = 0x0000;
  }
}

short dget_flag(int sh){
  return cdata[sh];
}

void put_data(char flag){
  rtf_put(1,(char *)data,0x4000);
  rtf_put(2,&flag,1);
}

void put_smem(void){
  cdata[0] = 0x0001;
}

int get_mp(void){
  return mp;
}

/* Test Function */
void print_shmem(int lines){
  int i,j;

  if(lines < 100){
    for(i=0;i<lines;i++){
      printk("%06o ",i*8);
      for(j=0;j<8;j++){
	printk("%04x ",data[i*8+j]);
      }
      printk("\n");
    }
  }
  printk("\n");
}

/* VME Series */
#ifdef VMEMM
int v785_segdata(unsigned long maddr){
  int wordcnt;
  unsigned long head;

  wordcnt = 0;
  vread32(maddr+V785_OUTBUFF,&head);

  if((data[mp-1] & V785_TYPE_MASK_S) == V785_HEADER_BIT_S){
    memcpy((char *)(data+mp),(char *)&head,4);
    mp += 2;
    segmentsize += 2;
    wordcnt++;
    while(wordcnt < 34){
      vread32(maddr+V785_OUTBUFF+wordcnt*32,(long *)(data+mp));
      mp += 2;
      segmentsize += 2;
      wordcnt++;
      if((data[mp-1] & (V785_TYPE_MASK_S)) != V785_DATA_BIT_S){
	break;
      }
    }
  }

  return segmentsize;
}

int v785_segmod(short segid,unsigned long maddr){
  int tmp,wordcnt;
  unsigned long head;

  wordcnt = 0;
  tmp = mp;
  mp++;
  data[mp++] = segid;

  vread32(maddr+V785_OUTBUFF,&head);

  if((data[mp-1] & V785_TYPE_MASK_S) == V785_HEADER_BIT_S){
    memcpy((char *)(data+mp),(char *)&head,4);
    mp += 2;
    wordcnt++;
    while(wordcnt < 34){
      vread32(maddr+V785_OUTBUFF+wordcnt*32,(long *)(data+mp));
      mp += 2;
      wordcnt++;
      if((data[mp-1] & (V785_TYPE_MASK_S)) != V785_DATA_BIT_S){
	break;
      }
    }
  }else{
    data[tmp] = 3;
    data[mp] = 0;
    eventsize += 3;

    return 3;
  }

  data[tmp] = wordcnt*2+2;                  /* Write word count */

  eventsize += wordcnt*2+2;                    /* Count up event size */

  return wordcnt*2 + 2;
}

#endif
