/*
 * CAEN V2718
 * PCI-VME
 * Hidetada Baba (RIKEN)
 * baba@ribf.riken.jp
 * 
 * last modified : 12/03/05 17:02:25 
 *
 */

// #define DEBUG
#ifdef DEBUG
#define DB(x) x
#else
#define DB(x)
#endif

#include "v2718.h"

int  v2718_init_register(int dn);
void v2718_reset(void);
void v2718_link_reset(void);
void v2718_set_amsr(unsigned char am);
void v2718_vread16(unsigned int addr,short *data);
void v2718_vread32(unsigned int addr,int *data);
void v2718_vwrite16(unsigned int addr,short *data);
void v2718_vwrite32(unsigned int addr,int *data);
void v2718_dma_vread32_start(unsigned int addr, int size);
int  v2718_dma_vread32_store(char *data, int size);
int  v2718_check_dma(void);
void v2718_init_dma(unsigned int addr,int rw,int ls, int size);
void v2718_dma_config(void);
void v2718_define_intlevel(int level);
void v2718_enable_interrupt(void);
void v2718_disable_interrupt(void);
int  v2718_check_interrupt(void);
int  v2718_read_intvector(void);
/* V2718 Front I/O */
int v2718_read_register(unsigned int addr);
void v2718_set_input_conf(int ch);
void v2718_set_output_conf(int ch);
void v2718_set_output_reg(int mask);
void v2718_clear_output_reg(int mask);

/* i series */
void v2718_set_amsr_i(unsigned char am, int dn);
void v2718_vread16_i(unsigned int addr,short *data, int dn);
void v2718_vread32_i(unsigned int addr,int *data, int dn);
void v2718_vwrite16_i(unsigned int addr,short *data, int dn);
void v2718_vwrite32_i(unsigned int addr,int *data, int dn);
int v2718_trstat(void);
int v2718_rxfifo(unsigned int addr);


#if LINUX_VERSION_CODE >=  0x020600
struct pci_config {
  resource_size_t addr[3];
  unsigned char irq;
}v2718config[16];
#else
struct pci_config {
  unsigned long addr[3];
  unsigned char irq;
}v2718config[16];
#endif

struct vpci_reg{
  char *csr;
  char *plx;
  char *dma_buff;
}vreg[16];


static int amsr[16];
volatile static int intlevel[16],devicen;

#ifdef DEBUG
volatile unsigned long long int RDTSC(void)
{
 unsigned int h,l;
  
 /* read Pentium cycle counter */
    __asm__(".byte 0x0f,0x31"
          :"=a" (l),
           "=d" (h));

  return ((unsigned long long int)h<<32)|l;
}
#endif

int v2718_init_register(int dn){
  unsigned bus_addr;
  int lval;

  vreg[dn].plx = ioremap(v2718config[dn].addr[0], V2718_PLX_SIZE);
  vreg[dn].csr = ioremap(v2718config[dn].addr[2], V2718_CSR_SIZE);
  vreg[dn].dma_buff = kmalloc(V2718_DMA_SIZE, GFP_KERNEL | GFP_DMA);
  if(vreg[dn].dma_buff <= 0){
    printk("V2718: Can't DAM Buffer was allocated.\n");
    return -1;
  }
  bus_addr = virt_to_bus(vreg[dn].dma_buff);
  writel(bus_addr, vreg[dn].plx + V2718_PLX_DMAPADR0);

  /* Reset */
  v2718_reset();
  udelay(1000); // sleep 1ms, to ensure the end of reset

  /* Enable Interrupt */
  lval = V2718_INTCSR_LOC_INT_ENA |  V2718_INTCSR_PCI_INT_ENA;
    // V2718_INTCSR_DMA0_INT_ENA |  V2718_INTCSR_DMA1_INT_ENA; // Dis DMA INT
  writel(lval, vreg[0].plx + V2718_PLX_INTCSR);

  writel(V2718_INTDIS, vreg[0].csr + V2718_IOCTL_SET);
  writel(V2718_VINTDIS, vreg[0].csr + V2718_IOCTL_SET);
  //writel(V2718_INTDIS, vreg[0].csr + V2718_IOCTL_CLR);

  v2718_dma_config();

  if(readl(vreg[dn].csr + V2718_STATUS) & V2718_LINK_FAIL ){
    printk("V2718: V2718 is power off or cable is not connected.\n");
  }
  

  v2718_set_amsr(A32);
  intlevel[dn] = 0;

  /* Initialize */
  writel(V2718_INIT1_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_INIT1_VALUE, vreg[0].csr + V2718_TXFIFO);
  v2718_rxfifo(0xffffffff);
  
  writel(V2718_INIT2_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_INIT2_VALUE, vreg[0].csr + V2718_TXFIFO);


  udelay(100);

  return 0;
}

void v2718_reset(void){
  unsigned int app;

  app = readl(vreg[0].plx + V2718_PLX_CNTRL);
  
  writel(app | V2718_CNTRL_SW_RESET, vreg[0].plx + V2718_PLX_CNTRL);
  udelay(500);
  writel(app & ~V2718_CNTRL_SW_RESET, vreg[0].plx + V2718_PLX_CNTRL);
  
  app = readl(vreg[0].plx + V2718_PLX_CNTRL);
  
  writel(app & ~V2718_CNTRL_CONF_RELOAD, vreg[0].plx + V2718_PLX_CNTRL);
  writel(app | V2718_CNTRL_CONF_RELOAD, vreg[0].plx + V2718_PLX_CNTRL);
  udelay(1000);
  v2718_link_reset();
}

void v2718_link_reset(void){
  writel(V2718_LNKRST, vreg[0].csr + V2718_IOCTL_CLR);
  udelay(1000);
  writel(V2718_LNKRST, vreg[0].csr + V2718_IOCTL_SET);
}

void v2718_set_amsr(unsigned char am){
  if(am == A16){
    amsr[0] = V2718_A16_BIT;
  }else if(am == A24 || am == A24BLK){
    amsr[0] = V2718_A24_BIT;
  }else{
    amsr[0] = V2718_A32_BIT;
  }
}

void v2718_set_vmeaddr(unsigned int addr){
}

int v2718_trstat(void){
  volatile int j;
  int ret;
  unsigned int lval;

  lval = 0xffffffff;
  ret = lval;
  for(j=0; j<100; j++){
    lval = readl(vreg[0].csr + V2718_TRSTAT);
    if(lval & 0xffff0000){
      ret = lval >> 16;
      if(ret > 12){
	j = 100;
      }else{
      }
    }
    udelay(1);
  }

  if(j < 100){
    lval = 0xffffffff;
    v2718_reset();
    printk("Can't oprate trstat");
  }

  return ret;
}  

int v2718_rxfifo(unsigned int addr){
  volatile int i, j;
  unsigned int lval;
  int size;

  lval = 0xffffffff;
  for(j=0; j<100; j++){
    lval = readl(vreg[0].csr + V2718_TRSTAT);
    if(lval & 0xffff0000){
      size = lval >> 16;
      for(i=0; i<size; i++){
	if(i == 1){
	  lval = readl(vreg[0].csr + V2718_RXFIFO);
	}else{
	  readl(vreg[0].csr + V2718_RXFIFO);
	}
      }
      j = 500;
    }
    udelay(1);
  }

  if(j < 500){
    lval = 0xffffffff;
    v2718_reset();
    printk("Can't oprate addr=0x%08x\n", (int)addr);
  }

  return lval;
}

void v2718_vread16(unsigned int addr,short *data){
  int lval;

  writel(V2718_R16_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_RD_BIT | amsr[0] | V2718_SINGLE_BIT | V2718_D16_BIT | 
	 (addr & 0x0000ffff) << 16, vreg[0].csr + V2718_TXFIFO);
  writel((addr & 0xffff0000) >> 16, vreg[0].csr + V2718_TXFIFO);

  lval = v2718_rxfifo(addr);

  *data = (short)(lval & 0x0000ffff);

}

void v2718_vread32(unsigned int addr,int *data){
  int lval;

  writel(V2718_R32_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_RD_BIT | amsr[0] | V2718_SINGLE_BIT | V2718_D32_BIT | 
	 (addr & 0x0000ffff) << 16, vreg[0].csr + V2718_TXFIFO);
  writel((addr & 0xffff0000) >> 16, vreg[0].csr + V2718_TXFIFO);

  lval = v2718_rxfifo(addr);

  *data = lval;

}

void v2718_vwrite16(unsigned int addr,short *data){

  writel(V2718_W16_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_WR_BIT | amsr[0] | V2718_SINGLE_BIT | V2718_D16_BIT | 
	 (addr & 0x0000ffff) << 16, vreg[0].csr + V2718_TXFIFO);
  writel(((*data << 16)& 0xffff0000) | ((addr & 0xffff0000) >> 16),
	 vreg[0].csr + V2718_TXFIFO);

  v2718_rxfifo(addr);
}

void v2718_vwrite32(unsigned int addr,int *data){
  writel(V2718_W32_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_WR_BIT | amsr[0] | V2718_SINGLE_BIT | V2718_D32_BIT | 
	 (addr & 0x0000ffff) << 16, vreg[0].csr + V2718_TXFIFO);
  writel(((*data << 16) & 0xffff0000) | ((addr & 0xffff0000) >> 16),
	 vreg[0].csr + V2718_TXFIFO);
  writel(((*data >> 16) & 0x0000ffff), vreg[0].csr + V2718_TXFIFO);

  v2718_rxfifo(addr);
}

void v2718_dma_vread32(unsigned int addr,int *data){
/*   offset_addr[0] = addr & V2718_OFFSET_MASK; */
/*   v2718_set_vmeaddr(addr); */
}

char *v2718_dma_addr(void){
  return vreg[0].dma_buff;
}

void v2718_dma_vread32_start(unsigned int addr,int size){
  int val, wsize;


  memset(vreg[0].dma_buff, 0, V2718_DMA_SIZE);
  
  writel(V2718_R32DMA_HEADER, vreg[0].csr + V2718_TXFIFO);
  val = ((size/4) << 16 & 0xffff0000) | 
    V2718_RD_BIT | amsr[0] | V2718_BLOCK_BIT | V2718_D32_BIT;

  writel((size << 16 & 0xffff0000) | V2718_RD_BIT | amsr[0] | V2718_BLOCK_BIT
	 | V2718_D32_BIT, vreg[0].csr + V2718_TXFIFO);
  writel(addr, vreg[0].csr + V2718_TXFIFO);
  wsize = v2718_trstat();
  if(wsize > 0 && wsize <= V2718_DMA_SIZE/4){
    wsize = wsize * 4;
    writel(wsize, vreg[0].plx + V2718_PLX_DMASIZE0);
    writel(V2718_DMACSR_ENA_0 | V2718_DMACSR_ENA_1| V2718_DMACSR_START_0,
	   vreg[0].plx + V2718_PLX_DMACSR);
    
  }else{
    printk("Can't start DMA (trstat error).\n");
  }

}

int v2718_dma_vread32_store(char *data, int size){
   int ret;

   if(v2718_check_dma()){
     ret = 1;
   }else{
     ret = 0;
     return ret;
   }

   readl(vreg[0].plx + V2718_PLX_DMASIZE0);

   memcpy(data, vreg[0].dma_buff+sizeof(int), size);
   writel(V2718_DMACSR_ENA_0 | V2718_DMACSR_ENA_1| V2718_DMACSR_CLRINT_0,
	  vreg[0].plx + V2718_PLX_DMACSR);


   return ret;
}

int v2718_check_dma(void){
  return readl(vreg[0].plx + V2718_PLX_DMACSR) & V2718_DMADONE_BIT;
}

int v2718_get_dma_status(void){
  //  return readb(vreg[dn].csr + V2718_LOCAL_STATUS);
  return 0;
}


void v2718_init_dma(unsigned int addr,int rw,int ls,int size){
/*   unsigned com,reg,remainder_cnt = 0,packet_cnt = 0; */

/* #ifdef DEBUG */
/*   volatile int ccc; */
/*   unsigned int long st,en; */
/* #endif */

/* #ifdef DEBUG */
/*   //memset(vreg[dn].dma_buff,1,V2718_DMA_SIZE); */
/*   st=RDTSC(); */
/* #endif */


/* #ifdef DEBUG */
/*   en=RDTSC(); */
/*   printk("ovf time %Ld\n",en-st); */
/*   st=RDTSC(); */
/*   en = 0; */
/*   outb(1,0x80); */
/*   for(ccc=0;ccc<1000;ccc++){ */
/*     if(v2718_check_dma_i(dn)){ */
/*       en=RDTSC(); */
/*       ccc = 10000; */
/*     }else{ */
/*       outb(1,0x80); */
/*     } */
/*   } */
/* #endif */

/* #ifdef DEBUG */
/*   printk("w2time %Ld\n",en-st); */
/* #endif */
}

void v2718_dma_config(void){
  unsigned int lval;
  
  /* Read */
  /*   lval = DMAMODE_DW32 | DMAMODE_READY_ENA |
     DMAMODE_BTERM_ENA | DMAMODE_BURST_ENA |
     DMAMODE_DONE_INT_ENA | DMAMODE_LOC_ADR_HOLD |
     DMAMODE_EOT_ENA | DMAMODE_INT_PCI | DMAMODE_FAST_TERM; */
  lval = V2718_DMAMODE_DW32 | V2718_DMAMODE_READY_ENA |
    V2718_DMAMODE_BTERM_ENA | V2718_DMAMODE_BURST_ENA |
    V2718_DMAMODE_LOC_ADR_HOLD | V2718_DMAMODE_EOT_ENA |
    V2718_DMAMODE_FAST_TERM;
  writel(lval, vreg[0].plx + V2718_PLX_DMAMODE0);
  writel(virt_to_bus(vreg[0].dma_buff), vreg[0].plx + V2718_PLX_DMAPADR0);
  writel(V2718_RXFIFO, vreg[0].plx + V2718_PLX_DMALADR0);
  writel(0xf, vreg[0].plx + V2718_PLX_DMADPR0);

 /* Write */
/*   lval = DMAMODE_DW32 | DMAMODE_READY_ENA | */
/*     DMAMODE_BTERM_ENA | DMAMODE_BURST_ENA | */
/*     DMAMODE_DONE_INT_ENA | DMAMODE_LOC_ADR_HOLD | */
/*     DMAMODE_EOT_ENA | DMAMODE_INT_PCI | DMAMODE_FAST_TERM; */
  lval = V2718_DMAMODE_DW32 | V2718_DMAMODE_READY_ENA | 
    V2718_DMAMODE_BTERM_ENA | V2718_DMAMODE_BURST_ENA |
    V2718_DMAMODE_LOC_ADR_HOLD | V2718_DMAMODE_EOT_ENA |
    V2718_DMAMODE_FAST_TERM;
  writel(lval, vreg[0].plx + V2718_PLX_DMAMODE1);
  writel(V2718_TXFIFO, vreg[0].plx + V2718_PLX_DMALADR1);
  writel(0x3, vreg[0].plx + V2718_PLX_DMADPR1);
}


void v2718_define_intlevel(int level){
  if(level > 0 && level < 8){
    intlevel[0] = level;
  }else{
    intlevel[0] = 0;
  }
}

void v2718_enable_interrupt(void){
  //writel(V2718_INTDIS, vreg[0].csr + V2718_IOCTL_CLR);
  writel(V2718_VINTDIS, vreg[0].csr + V2718_IOCTL_CLR);
}

void v2718_disable_interrupt(void){
  //writel(V2718_INTDIS, vreg[0].csr + V2718_IOCTL_SET);
  writel(V2718_VINTDIS, vreg[0].csr + V2718_IOCTL_SET);

}

int v2718_check_interrupt(void){
  unsigned int intcsr, irq0, irq1;

  writel(V2718_CHKIRQ_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_CHKIRQ_VALUE, vreg[0].csr + V2718_TXFIFO);
  v2718_rxfifo(0xffffffff);

  intcsr = readl(vreg[0].plx + V2718_PLX_INTCSR);
  if(intcsr & V2718_INTCSR_LOC_INT_ACT){

    irq0 = readl(vreg[0].csr + V2718_IRQSTAT0);
    irq1 = readl(vreg[0].csr + V2718_IRQSTAT1);
    intcsr = readl(vreg[0].csr + V2718_IRQSLAVES);

    return 1;
  }

  return 0;
}

int v2718_read_intvector(void){
  /* Acknowledge intterupt */
  writel(V2718_ACKIRQ_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_ACKIRQ_BIT1 | (intlevel[0] << 17), vreg[0].csr + V2718_TXFIFO);
  writel(V2718_ACKIRQ_BIT2, vreg[0].csr + V2718_TXFIFO);
  v2718_rxfifo(0xffffffff);

  return 0;
}

/*  V2718 internal register */
int v2718_read_register(unsigned int addr){
  int val;

  writel(V2718_READREG_HEADER, vreg[0].csr + V2718_TXFIFO);
  writel(V2718_READREG_VALUE | addr, vreg[0].csr + V2718_TXFIFO);
  val = v2718_rxfifo(0xffffffff);
  printk("read reg val = 0x%08x\n", val);
  
  return val;
}

void v2718_set_input_conf(int ch){
}

void v2718_set_output_conf(int ch){
  int tval;

  writel(V2718_SETOUTPUTCONF_HEADER, vreg[0].csr + V2718_TXFIFO);
  tval = (0x3 << ch*2) | (0x1 << 10);
  tval = tval << 16 | V2718_CLR_OUTPUT_MULTI;
  writel(tval, vreg[0].csr + V2718_TXFIFO);

  writel(V2718_SETOUTPUTCONF_HEADER, vreg[0].csr + V2718_TXFIFO);
  tval = (0x3 << ch*2);
  tval = tval << 16 | V2718_SET_OUTPUT_MULTI;
  writel(tval, vreg[0].csr + V2718_TXFIFO);
}
void v2718_set_output_reg(int mask){
  int tval;

  writel(V2718_SETOUTPUTREG_HEADER, vreg[0].csr + V2718_TXFIFO);
  tval = (mask << 6) << 16 | V2718_SET_OUTPUT_REG;
  writel(tval, vreg[0].csr + V2718_TXFIFO);
}

void v2718_clear_output_reg(int mask){
  int tval;

  writel(V2718_SETOUTPUTREG_HEADER, vreg[0].csr + V2718_TXFIFO);
  tval = (mask << 6) << 16 | V2718_CLR_OUTPUT_REG;
  writel(tval, vreg[0].csr + V2718_TXFIFO);
}

/* i series */
void v2718_set_amsr_i(unsigned char am, int dn){
  v2718_set_amsr(am);
}

void v2718_vread16_i(unsigned int addr,short *data, int dn){
  v2718_vread16(addr, data);
}

void v2718_vread32_i(unsigned int addr,int *data, int dn){
  v2718_vread32(addr, data);
}

void v2718_vwrite16_i(unsigned int addr,short *data, int dn){
  v2718_vwrite16(addr, data);
}

void v2718_vwrite32_i(unsigned int addr,int *data, int dn){
  v2718_vwrite32(addr, data);
}
