/*
 * SBS620
 * PCI-VME
 * 
 * Hidetada Baba
 * Aug 8, 2005
 */

//#define DEBUG

#include "sbs620.h"

int sbs620_init_register(int dn);
void sbs620_set_amsr(unsigned char am);
void sbs620_vread16(unsigned int addr,short *data);
void sbs620_vread32(unsigned int addr,int *data);
void sbs620_vwrite16(unsigned int addr,short *data);
void sbs620_vwrite32(unsigned int addr,int *data);
void sbs620_dma_vread32_start(unsigned int addr, int size);
int sbs620_dma_vread32_store(char *data, int size);
int sbs620_check_dma(void);
void sbs620_init_dma(unsigned int addr,int rw,int ls, int size);
void sbs620_define_intlevel(int level);
void sbs620_enable_interrupt(void);
void sbs620_disable_interrupt(void);
int sbs620_check_interrupt(void);
int sbs620_read_intvector(void);


void sbs620_set_amsr_i(unsigned char am,int dn);
void sbs620_vread16_i(unsigned int addr,short *data,int dn);
void sbs620_vread32_i(unsigned int addr,int *data,int dn);
void sbs620_vwrite16_i(unsigned int addr,short *data,int dn);
void sbs620_vwrite32_i(unsigned int addr,int *data,int dn);
void sbs620_dma_vread32_start_i(unsigned int addr,int size, int dn);
int sbs620_dma_vread32_store_i(char *data,int size,int dn);
int sbs620_check_dma_i(int dn);
void sbs620_init_dma_i(unsigned int addr,int rw,int ls,int size,int dn);
void sbs620_define_intlevel_i(int level,int dn);
void sbs620_enable_interrupt_i(int dn);
void sbs620_disable_interrupt_i(int dn);
int sbs620_check_interrupt_i(int dn);

#if LINUX_VERSION_CODE >= 0x020600
struct pci_config {
  resource_size_t addr[4];
  unsigned char irq;
}sbsconfig[16];
#else
struct pci_config {
  unsigned long addr[4];
  unsigned char irq;
}sbsconfig[16];
#endif

struct vpci_reg{
  char *io;
  char *csr;
  char *vme;
  char *pci;
  char *dma;
  char *remote;
  char *dma_buff;
}vreg[16];


static int amsr[16];
#if LINUX_VERSION_CODE >= 0x020600
static resource_size_t base_addr[16],offset_addr[16];
#else
static unsigned long base_addr[16],offset_addr[16];
#endif

volatile static int intlevel[16],devicen;
volatile static short intleveln = 0;

#ifdef DEBUG
volatile unsigned long long int RDTSC(void)
{
 unsigned int h,l;
  
 /* read Pentium cycle counter */
    __asm__(".byte 0x0f,0x31"
          :"=a" (l),
           "=d" (h));

  return ((unsigned long long int)h<<32)|l;
}
#endif


int sbs620_init_register(int dn){
  unsigned char sta;
  unsigned bus_addr;
  int i;

  vreg[dn].csr = ioremap(sbsconfig[dn].addr[1],SBS620_CSR_SIZE);
  vreg[dn].vme = ioremap(sbsconfig[dn].addr[2],SBS620_MAP_SIZE);
  vreg[dn].pci = vreg[dn].vme + SBS620_PCI_OFFSET;
  vreg[dn].dma = vreg[dn].vme + SBS620_DMA_OFFSET;
  vreg[dn].remote = ioremap(sbsconfig[dn].addr[3],SBS620_REMOTE_SIZE);

  vreg[dn].dma_buff = kmalloc(SBS620_DMA_SIZE,GFP_KERNEL|GFP_DMA);
  if(vreg[dn].dma_buff <= 0){
    printk("Can't DAM Buffer was allocated.\n");
    return -1;
  }
  bus_addr = virt_to_bus(vreg[dn].dma_buff) & SBS620_DMA_BASE_MASK;
    for (i=0; i<SBS620_DMA_SIZE/VME_PAGE; i++) {
      writel(bus_addr+VME_PAGE*i,vreg[dn].dma+i*4);
      //    writel(bus_addr,vreg[dn].dma);
      printk("%d %x\n",i,bus_addr+VME_PAGE*i);
    }

  sta = readb(vreg[dn].csr+SBS620_LOCAL_STATUS) & SBS620_POWER;
 
  if(sta > 0){
    printk("SBS620 is power off or cable is not connected.\n");
    /* return -1; */
  }

  writeb(SBS620_DMA_BLOCK_MODE,vreg[dn].csr + SBS620_REMOTE_COMMAND2);
  
  sbs620_set_amsr_i(A32,dn);
  intlevel[dn] = 0;

  return 0;
}

void sbs620_set_amsr(unsigned char am){
  amsr[0] = ((am << SBS620_AM_SHIFT) | 
	     (SBS620_BUS_IO << SBS620_FUNC_SHIFT)|
	     (SBS620_SWAP_NONE << SBS620_SWAP_SHIFT));
}

void sbs620_set_vmeaddr(unsigned int addr){
  base_addr[0] = addr & SBS620_BASE_MASK;
  writel(base_addr[0] | amsr[0], vreg[0].vme);
}

void sbs620_vread16(unsigned int addr,short *data){

  offset_addr[0] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr(addr);

  *data = readw(vreg[0].remote + offset_addr[0]);
}

void sbs620_vread32(unsigned int addr,int *data){

  offset_addr[0] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr(addr);

  *data = readl(vreg[0].remote + offset_addr[0]);
}

void sbs620_vwrite16(unsigned int addr,short *data){
  offset_addr[0] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr(addr);

  writew(*data,vreg[0].remote + offset_addr[0]);

}

void sbs620_vwrite32(unsigned int addr,int *data){
  offset_addr[0] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr(addr);

  writel(*data,vreg[0].remote + offset_addr[0]);

}

void sbs620_dma_vread32(unsigned int addr,int *data){
  offset_addr[0] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr(addr);
  
}

char *sbs620_dma_addr(void){
  return vreg[0].dma_buff;
}

void sbs620_dma_vread32_start_i(unsigned int addr,int size, int dn){
  sbs620_init_dma_i(addr,SBS620_DMA_READ,SBS620_DMA_LONGWORD,size,dn);
}

int sbs620_dma_vread32_store_i(char *data, int size, int dn){
  int ret;

  if(sbs620_check_dma_i(dn)){
    ret = 1;
  }else{
    ret = 0;
    return ret;
  }

  memcpy(data,vreg[dn].dma_buff,size);

  writeb(0,vreg[dn].csr + SBS620_DMA_COMMAND);
  writeb(SBS620_DMA_BLOCK_MODE,vreg[dn].csr + SBS620_REMOTE_COMMAND2);

  return ret;
}

int sbs620_check_dma_i(int dn){
  return readb(vreg[dn].csr + SBS620_DMA_COMMAND) & SBS620_DMA_DONE;
}

int sbs620_get_dma_status_i(int dn){
  return readb(vreg[dn].csr + SBS620_LOCAL_STATUS);
}


void sbs620_init_dma_i(unsigned int addr,int rw,int ls,int size, int dn){
  unsigned com,reg,remainder_cnt = 0,packet_cnt = 0;

#ifdef DEBUG
  volatile int ccc;
  unsigned long long st,en;
#endif

#ifdef DEBUG
  //memset(vreg[dn].dma_buff,1,SBS620_DMA_SIZE);
  st=RDTSC();
#endif


  writeb(0,vreg[dn].csr + SBS620_DMA_COMMAND);
  com = 0;
  com = rw | ls;
  writeb(com,vreg[dn].csr + SBS620_DMA_COMMAND);

  reg = 0;
  writeb(reg,vreg[dn].csr + SBS620_DMA_PCI_ADDR_2_7);
  writeb(reg,vreg[dn].csr + SBS620_DMA_PCI_ADDR_8_15);
  writeb(reg,vreg[dn].csr + SBS620_DMA_PCI_ADDR_16_23);

  writeb((addr) & 0x000000ff,vreg[dn].csr + SBS620_DMA_VME_ADDR_0_7);
  writeb((addr >> 8) & 0x000000ff,vreg[dn].csr + SBS620_DMA_VME_ADDR_8_15);
  writeb((addr >> 16) & 0x000000ff,vreg[dn].csr + SBS620_DMA_VME_ADDR_16_23);
  writeb((addr >> 24) & 0x000000ff,vreg[dn].csr + SBS620_DMA_VME_ADDR_24_31);

  /* reg = SBS620_ENABLE_INTERRUPT; */
  reg = 0; // Not use interrupt for DMA end
  writeb(reg,vreg[dn].csr + SBS620_INT_CTRL);

  remainder_cnt = size % SBS620_DMA_PACKET_SIZE;
  packet_cnt    = size / SBS620_DMA_PACKET_SIZE;
  writeb(remainder_cnt,vreg[dn].csr + SBS620_DMA_REMAINDER_CNT);
  writeb(remainder_cnt,vreg[dn].csr + SBS620_DMA_REMOTE_REMAINDER_CNT);
  writeb((packet_cnt) & 0x00ff,vreg[dn].csr + SBS620_DMA_PACKET_CNT_0_7);
  writeb((packet_cnt >> 8) & 0x00ff,vreg[dn].csr + SBS620_DMA_PACKET_CNT_8_15);
  
  //reg = SBS620_DMA_BLOCK_MODE | SBS620_DMA_DIS_INT_PASS | SBS620_DMA_PAUSE_ON;
  reg = SBS620_DMA_BLOCK_MODE | SBS620_DMA_DIS_INT_PASS;
  //reg = SBS620_DMA_DIS_INT_PASS;
  writeb(reg,vreg[dn].csr + SBS620_REMOTE_COMMAND2);

  reg = 0x0b; // A32 Block
  /* reg = 0x3b; */ // A24 Block
  //reg = 0x09;
  writeb(reg,vreg[dn].csr + SBS620_REMOTE_AM);

  com |= SBS620_DMA_STARTDMA;
  writeb(com,vreg[dn].csr + SBS620_DMA_COMMAND);

#ifdef DEBUG
  en=RDTSC();
  printk("ovf time %Ld\n",en-st);
  st=RDTSC();
  en = 0;
  outb(1,0x80);
  for(ccc=0;ccc<1000;ccc++){
    if(sbs620_check_dma_i(dn)){
      en=RDTSC();
      ccc = 10000;
    }else{
      outb(1,0x80);
    }
  }
#endif

#ifdef DEBUG
  printk("w2time %Ld\n",en-st);
#endif
}

void sbs620_define_intlevel(int level){
  intleveln = (short)level;
  intlevel[0] = 1<<level;
}

void sbs620_enable_interrupt(void){
  writeb(SBS620_CLEAR_INTERRUPT,vreg[0].csr + SBS620_LOCAL_COMMAND);
  writeb(SBS620_ENABLE_INTERRUPT,vreg[0].csr + SBS620_INT_CTRL);
}
  
void sbs620_disable_interrupt(void){
  writeb(SBS620_DISABLE_INTERRUPT,vreg[0].csr + SBS620_INT_CTRL);
  writeb(SBS620_CLEAR_INTERRUPT,vreg[0].csr + SBS620_LOCAL_COMMAND);
}

int sbs620_check_interrupt(void){
  if((readb(vreg[0].csr + SBS620_INT_CTRL) & SBS620_IS_INTERRUPT)
     == SBS620_IS_INTERRUPT){
    if((readb(vreg[0].csr + SBS620_INT_STATUS) & intlevel[0]) == intlevel[0]){
      return 1;
    }
  }

  return 0;
}

int sbs620_read_intvector(void){
  writew(intleveln,vreg[0].csr + SBS620_REMOTE_COMMAND1);
  return readw(vreg[0].csr + SBS620_IACK_LOW);
}


 
/* i series */

void sbs620_set_amsr_i(unsigned char am,int dn){
  amsr[dn] = ((am << SBS620_AM_SHIFT) | 
	     (SBS620_BUS_IO << SBS620_FUNC_SHIFT)|
	     (SBS620_SWAP_NONE << SBS620_SWAP_SHIFT));
}

void sbs620_set_vmeaddr_i(unsigned int addr,int dn){
  base_addr[dn] = addr & SBS620_BASE_MASK;
  writel(base_addr[dn] | amsr[dn], vreg[dn].vme);
}

void sbs620_vread16_i(unsigned int addr,short *data,int dn){

  offset_addr[dn] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr_i(addr,dn);

  *data = readw(vreg[dn].remote + offset_addr[dn]);
}

void sbs620_vread32_i(unsigned int addr,int *data,int dn){

  offset_addr[dn] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr_i(addr,dn);

  *data = readl(vreg[dn].remote + offset_addr[dn]);
}

void sbs620_vwrite16_i(unsigned int addr,short *data,int dn){
  offset_addr[dn] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr_i(addr,dn);

  writew(*data,vreg[dn].remote + offset_addr[dn]);

}

void sbs620_vwrite32_i(unsigned int addr,int *data,int dn){
  offset_addr[dn] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr_i(addr,dn);

  writel(*data,vreg[dn].remote + offset_addr[dn]);

}

void sbs620_dma_vread32_i(unsigned int addr,int *data,int dn){
  offset_addr[dn] = addr & SBS620_OFFSET_MASK;
  sbs620_set_vmeaddr_i(addr,dn);
  
}

void sbs620_dma_vread32_start(unsigned int addr, int size){
  sbs620_dma_vread32_start_i(addr,size,0);
}

int sbs620_dma_vread32_store(char *data,int size){
  return sbs620_dma_vread32_store_i(data,size,0);
}

int sbs620_check_dma(void){
  return sbs620_check_dma_i(0);
}

void sbs620_init_dma(unsigned int addr,int rw,int ls,int size){
  sbs620_init_dma_i(addr,rw,ls,size,0);
}

void sbs620_define_intlevel_i(int level,int dn){
  intlevel[dn] = 1<<level;
}

void sbs620_enable_interrupt_i(int dn){
  writeb(SBS620_CLEAR_INTERRUPT,vreg[dn].csr + SBS620_LOCAL_COMMAND);
  writeb(SBS620_ENABLE_INTERRUPT,vreg[dn].csr + SBS620_INT_CTRL);
}
  
void sbs620_disable_interrupt_i(int dn){
  writeb(SBS620_DISABLE_INTERRUPT,vreg[dn].csr + SBS620_INT_CTRL);
  writeb(SBS620_CLEAR_INTERRUPT,vreg[dn].csr + SBS620_LOCAL_COMMAND);
}

int sbs620_check_interrupt_i(int dn){
  if((readb(vreg[dn].csr + SBS620_INT_CTRL) & SBS620_IS_INTERRUPT)
     == SBS620_IS_INTERRUPT){
    if((readb(vreg[dn].csr + SBS620_INT_STATUS) & intlevel[dn]) == intlevel[dn]){
      return 1;
    }
  }

  return 0;
}
