/*
 * SBS620 PCI-VME
 * VME Access Module
 * Hidetada Baba (RIKEN)
 * baba@ribf.riken.jp
 * May 29, 2008
 */

#include <linux/version.h>
#if LINUX_VERSION_CODE >=  0x020600
#if defined(USE_MODVERSIONS) && USE_MODVERSIONS
#  define MODVERSIONS
#  include <config/modversions.h>
#endif
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#if LINUX_VERSION_CODE >=  0x020600
#include <linux/init.h>
#endif
#include <linux/ioctl.h>
#include <linux/sched.h>
#if LINUX_VERSION_CODE <  0x020600
#include <linux/config.h>
#endif
#include <linux/pci.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/delay.h>

#if LINUX_VERSION_CODE <  0x020400
#include <linux/malloc.h>
#endif

#if LINUX_VERSION_CODE >=  0x020410
MODULE_LICENSE("GPL");
#endif
#if LINUX_VERSION_CODE >=  0x020600
MODULE_AUTHOR("Hidetada Baba");
#endif

#include "sbs620.c"


int sbs620_get_irq(void);

#if LINUX_VERSION_CODE >= 0x020600
struct pci_dev* sbs620_pci_dev;

static int sbs620_pci_probe(struct pci_dev *dev, const struct pci_device_id *id);
static void sbs620_pci_remove(struct pci_dev *dev);

static struct pci_driver sbs620_pci_driver = {
  .id_table = sbs620_device_id_table,
  .probe = sbs620_pci_probe,
  .remove = sbs620_pci_remove,
};
static volatile int probflag;
#else
int detect_pci_device(unsigned vendor,unsigned device,struct pci_config *config,int *devn);
#endif

#if LINUX_VERSION_CODE >= 0x020600
EXPORT_SYMBOL(sbs620_set_amsr);
EXPORT_SYMBOL(sbs620_vread16);
EXPORT_SYMBOL(sbs620_vread32);
EXPORT_SYMBOL(sbs620_vwrite16);
EXPORT_SYMBOL(sbs620_vwrite32);
EXPORT_SYMBOL(sbs620_get_irq);
EXPORT_SYMBOL(sbs620_define_intlevel);
EXPORT_SYMBOL(sbs620_enable_interrupt);
EXPORT_SYMBOL(sbs620_disable_interrupt);
EXPORT_SYMBOL(sbs620_check_interrupt);
EXPORT_SYMBOL(sbs620_dma_vread32_start);
EXPORT_SYMBOL(sbs620_dma_vread32_store);
EXPORT_SYMBOL(sbs620_read_intvector);
#endif



#if LINUX_VERSION_CODE >= 0x020600
static int sbs620_init_module(void)
#else
int init_module(void)
#endif
{
  int ret,i, err;
  devicen = 0;
  
  ret = 0;
#if LINUX_VERSION_CODE >= 0x020600
  sbs620_pci_driver.name = "SBS620";
  sbs620_pci_dev = NULL;
  probflag = 0;
  err = pci_register_driver(&sbs620_pci_driver);
  if(sbs620_pci_dev == NULL || err < 0){
    printk("SBS620 Can't find.\n");
    return -ENODEV;
  }
  /* Register again to avoid APIC's IRQ change */
  schedule_timeout(5);
  probflag = 1;
  pci_unregister_driver(&sbs620_pci_driver);
  err = pci_register_driver(&sbs620_pci_driver);
#else
  ret = detect_pci_device(SBS620_VENDOR_ID,SBS620_DEVICE_ID,(struct pci_config *)&sbsconfig,(int *)&devicen);
  if(ret){
    printk("SBS620 Can't find.\n");
    return ret;
  }
#endif
  for(i=0;i<devicen;i++){
    printk("%d : SBS620 found at 0x%08x 0x%08x 0x%08x 0x%08x on irq %d.\n",
	   i,sbsconfig[i].addr[0],sbsconfig[i].addr[1],
	   sbsconfig[i].addr[2],sbsconfig[i].addr[3],
	   sbsconfig[i].irq);
    if(sbs620_init_register(i) == -1){
      return -ENODEV;
    }
  }
  
  return 0;
}

#if LINUX_VERSION_CODE >= 0x020600
static void sbs620_cleanup_module(void)
#else
void cleanup_module(void)
#endif
{
  int i;

  for(i=0;i<devicen;i++){
    kfree(vreg[i].dma_buff);
    vfree(vreg[i].csr);
    vfree(vreg[i].vme);
    vfree(vreg[i].remote);
  }

#if LINUX_VERSION_CODE >= 0x020600
  pci_unregister_driver(&sbs620_pci_driver);
#endif


}

#if LINUX_VERSION_CODE < 0x020600
int detect_pci_device(unsigned vendor,unsigned device,struct pci_config *sbsconfig,int *devicen){
  struct pci_dev *dev;
  int i,j;

  if (!pcibios_present()){
    printk("SBS620: unable to find PCI bios.\n");
    return -ENODEV;
  }
  dev = NULL;
  for(j=0;j<16;j++){
    dev = pci_find_device(vendor, device, dev);
    if(dev == NULL){
      break;
    }
#if LINUX_VERSION_CODE >=  0x020400
      for(i=0;i<4;i++){
	sbsconfig[j].addr[i] = dev->resource[i].start & PCI_BASE_ADDRESS_IO_MASK;
      }
      sbsconfig[j].irq = dev->irq;
#else
      for(i=0;i<4;i++){
	sbsconfig[j].addr[i] = dev->base_address[i] & PCI_BASE_ADDRESS_IO_MASK;
      }
      sbsconfig[j].irq = dev->irq;
#endif
      
  }
  
  if(j==0){
    return -ENODEV;
  }
  
  *devicen = j;
  
  return 0;
}
#endif

int sbs620_get_irq(void){
  return sbsconfig[0].irq;
}


#if LINUX_VERSION_CODE >= 0x020600
static int sbs620_pci_probe(struct pci_dev *dev, const struct pci_device_id *id){
  int i,j, ret;

  sbs620_pci_dev = dev;

  // probflag -> to avoid counting devicen by the dummy probe actions
  if(probflag){
    j = devicen;
    for(i=0;i<4;i++){
      sbsconfig[j].addr[i] = dev->resource[i].start & PCI_BASE_ADDRESS_IO_MASK;
    }
    sbsconfig[j].irq = dev->irq;
    devicen++;   // Count the number of PCI devices
  }

  ret = pci_enable_device(dev);
  return ret;
}

static void sbs620_pci_remove(struct pci_dev *dev){
}


module_init(sbs620_init_module);
module_exit(sbs620_cleanup_module);
#endif
