/*
 * Kinetic 2915 PCI Interface to 3922 Controller
 * 
 * H.baba
 * May 13, 2002
 */

#include "k2915.h"

void k2915_init_register(int irq,int ior,int pbor);
int k2915_check_done(int try);
int k2915_check_emp_fifo(int mtry);
int k2915_check_full_fifo(void);
int k2915_check_lam(void);
int k2915_control(int cnaf);
int k2915_read16(int cnaf,short *data);
int k2915_read24(int cnaf,short *data);
int k2915_write16(int cnaf,short *data);
int k2915_write24(int cnaf,int *data);
int k2915_block_read16(int mode,int cnaf,short *data,int count);
int k2915_block_read24(int mode,int cnaf,unsigned int *data,int count);
void k2915_crate_reset(void);
void k2915_rfs_enable_interrupt(void);
void k2915_rfs_disable_interrupt(void);
void k2915_pci_enable_interrupt(void);
void k2915_pci_clear_interrupt(void);
void k2915_crate_enable_lam(int lammask);
void k2915_crate_define_lam(int n);
void k2915_crate_disable_lam(void);
void k2915_crate_z(int c);
void k2915_crate_c(int c);
void k2915_crate_seti(int c);
void k2915_crate_deli(int c);
void k2915_get_csrdata(int *ret);
void k2915_get_bmcsdata(int *ret);
void k2915_read_lam(int *data);
int k2915_dma_block_read16(int mode,int cnaf,int count);
int k2915_dma_block_read24(int mode,int cnaf,int count);


static int try,done_try;

struct pci_struct{
  int base1;
  int base2;
  unsigned int irq;
};


struct pci_struct pci_table[] ={
  {0, 0, 0}
};

struct pci_reg{
  unsigned int csr;
  unsigned int cnaf;
  unsigned int tcr;
  unsigned int srr;
  unsigned int fifo;
  unsigned int mwa;
  unsigned int mwtc;
  unsigned int mra;
  unsigned int mrtc;
  unsigned int ics;
  unsigned int bmcs;  
};

struct pci_reg reg[] ={
  {CSR, CNAF, TCR, SRR, FIFO, MWA, MWTC, MRA, MRTC, ICS, BMCS,},
};

void k2915_init_register(int irq,int ior,int pbor){

  pci_table->irq   = (unsigned int)irq;
  pci_table->base1 = (unsigned int)ior;
  pci_table->base2 = (unsigned int)pbor;
  
  reg->csr  += (unsigned int)pbor;
  reg->cnaf += (unsigned int)pbor;
  reg->tcr  += (unsigned int)pbor;
  reg->srr  += (unsigned int)pbor;

  reg->fifo += (unsigned int)ior;
  reg->mwa  += (unsigned int)ior;
  reg->mwtc += (unsigned int)ior;
  reg->mra  += (unsigned int)ior;
  reg->mrtc += (unsigned int)ior;
  reg->ics  += (unsigned int)ior;
  reg->bmcs += (unsigned int)ior;

}

int k2915_check_done(int mtry){
  int ret = 0;

  done_try = 0;
  while (!(inl(reg->csr) & DONE)){
    done_try++;
    if(done_try >= mtry) {
      if(mtry == MAX_DONE_TRY){
	printk("check_done: Timeout, Waiting for DONE\n");
      }
      ret = 1;
      break;
    }
  }
  return ret;
}

int k2915_check_emp_fifo(int mtry){
  int ret = 0;

  try = 0;
  while((inl(reg->bmcs) & IN_FIFO_EMPTY)){
    try++;
    if(try >= mtry) {
      if(mtry == MAX_FIFO_TRY){
	printk("check_emp_fifo: Timeout, inbound FIFO empty\n");
      }
      ret = 1;
      break;
    }
  }
  return ret;
}

int k2915_check_full_fifo(void){
  int ret = 0;

  try = 0;
  while((inl(reg->bmcs) & OUT_FIFO_FULL)){
    try++;
    if(try > MAX_FIFO_TRY) {
      /* printk("check_full_fifo: Timeout, outbound FIFO full\n"); */
      ret = 1;
      break;
    }
  }
  return ret;
}

int k2915_check_lam(void){
  return (inl(reg->csr) & RFS);
}

void k2915_read_lam(int *data){
  *data =  (inl(reg->csr) & RFS);
}

int k2915_control(int cnaf){
  outl(cnaf, reg->cnaf);
  outl((unsigned int)GO,reg->csr);
  k2915_check_done(MAX_DONE_TRY);
  return 1;
}

int k2915_read16(int cnaf,short *data){
  outl((unsigned int)FIFO_RESET,reg->bmcs); 
  outl(cnaf,reg->cnaf);
  outl((unsigned int)(SINGLE16 | GO),reg->csr);
  /* check_done(MAX_DONE_TRY); */
  if(k2915_check_emp_fifo(MAX_FIFO_TRY)) return 0;
  *data = inl(reg->fifo);
  return 1;
}

int k2915_read24(int cnaf,short *data){
  unsigned int tdata;
  outl((unsigned int)FIFO_RESET,reg->bmcs); 
  outl(cnaf,reg->cnaf);
  outl((unsigned int)(SINGLE24 | GO),reg->csr);
  /* check_done(MAX_DONE_TRY); */
  if(k2915_check_emp_fifo(MAX_DONE_TRY)) return 0;
  tdata = inl(reg->fifo);
  *data = tdata & 0xffff;
  *(data+1) = (tdata>>16) & 0x00ff;
  return 1;
}

int k2915_write16(int cnaf,short *data){
  /* outl((unsigned int)FIFO_RESET,reg->bmcs); */
  outl(cnaf,reg->cnaf);
  /* check_full_fifo(); */
  outl((unsigned int)*data,reg->fifo);
  outl((unsigned int)(SINGLE16 | GO),reg->csr);
  k2915_check_done(MAX_DONE_TRY);
  return 1;
}

int k2915_write24(int cnaf,int *data){
  /* outl((unsigned int)FIFO_RESET,reg->bmcs); */
  outl(cnaf,reg->cnaf);
  /* check_full_fifo(); */
  outl((unsigned int)*data,reg->fifo);
  outl((unsigned int)(SINGLE24 | GO),reg->csr);
  k2915_check_done(MAX_DONE_TRY);
  return 1;
}

int k2915_block_read16(int mode,int cnaf,short *data,int count){
  int ret = 0,btry = 0;
  unsigned int tdata;

  outl(cnaf,reg->cnaf);
  outl(~(count-1),reg->tcr);
  outl((unsigned int)FIFO_RESET,reg->bmcs); 
  outl((unsigned int)(GO | SINGLE16 | mode),reg->csr);

  while(!k2915_check_emp_fifo(MAX_FIFO_TRY)){
    tdata = inl(reg->fifo);
    *(data+btry) = tdata & 0xffff;
    btry++;
    *(data+btry) = (tdata >> 16)& 0xffff;
    btry++;
    if(btry >= count){
      /* printk("btry = count\n"); */
      break;
    }
    if(!k2915_check_done(1)){
      if(!k2915_check_emp_fifo(1)){
	tdata = inl(reg->fifo);
	*(data+btry) = tdata & 0xffff;
	btry++;
	*(data+btry) = (tdata >> 16)& 0xffff;
	btry++;
      }
      /* printk("DONE\n"); */
      break;
    }
    /* outl((unsigned int)FIFO_RESET,reg->bmcs); */
  }

  /*
    if(inl(reg->csr) & ERR){
    ret = -E_CC_DMA;
    }
  */

  ret = btry;

  return ret;
}

int k2915_dma_block_read16(int mode,int cnaf,int count){
  return 0;
}

int k2915_dma_block_read24(int mode,int cnaf,int count){
  return 0;
}


/*
# short *data  --> unsigned int *data

  int block_read24(int mode,int cnaf,short *data,int count){
  int ret = 0,btry = 0;
  unsigned int tdata;

  outl(cnaf,reg->cnaf);
  outl(~(count-1),reg->tcr);
  outl((unsigned int)FIFO_RESET,reg->bmcs); 
  outl((unsigned int)(GO | SINGLE24 | mode),reg->csr);

  count *= 2;

  while(!check_emp_fifo()){
    tdata = inl(reg->fifo);
    *(data+btry) = tdata & 0xffff;
    btry++;
    *(data+btry) = (tdata >> 16)& 0x00ff;
    btry++;

    if(btry >= count){
      break;
    }
    if(!check_done(1)){
      tdata = inl(reg->fifo);
      *(data+btry) = tdata & 0xffff;
      btry++;
      *(data+btry) = (tdata >> 16)& 0x00ff;
      btry++;
      break;
    }
  }
*/

int k2915_block_read24(int mode,int cnaf,unsigned int *data,int count){
  int ret = 0,btry = 0;

  outl(cnaf,reg->cnaf);
  outl(~(count-1),reg->tcr);
  outl((unsigned int)FIFO_RESET,reg->bmcs); 
  outl((unsigned int)(GO | SINGLE24 | mode),reg->csr);

  while(!k2915_check_emp_fifo(MAX_FIFO_TRY)){
    *(data+btry) = inl(reg->fifo) & 0x00ffffff;
    btry++;

    if(btry >= count){
      /* printk("btry = count\n"); */
      break;
    }
    if(!k2915_check_done(1)){
      if(!k2915_check_emp_fifo(1)){
	*(data+btry) = inl(reg->fifo) & 0x00ffffff;
	btry++;
      }
      /* printk("DONE\n"); */
      break;
    }
    /* outl((unsigned int)FIFO_RESET,reg->bmcs); */
  }

  /*
    if(inl(reg->csr) & ERR){
    ret = -E_CC_DMA;
    }
  */

  ret = btry;

  return ret;
}


void k2915_crate_reset(void){
  short i = CLR;

  outl((unsigned int)RST_INFC,reg->csr);
  outl((unsigned int)(CLR_PCII | inl(reg->csr)), reg->csr);
  outl((unsigned int)(FIFO_RESET | inl(reg->bmcs)),reg->bmcs);
   
  k2915_check_done(MAX_DONE_TRY);
  k2915_write16(NAFGEN(30, 0, 17), &i);
}

void k2915_rfs_enable_interrupt(void){
  outl((unsigned int)RFS_IENA,reg->csr);
}

void k2915_rfs_disable_interrupt(void){
  outl((unsigned int)(~RFS_IENA & (inl(reg->csr) & RFS_MASK)),reg->csr);
}

void k2915_pci_enable_interrupt(void){
  outl((unsigned int)PCI_IENA,reg->csr);
}

void k2915_pci_clear_interrupt(void){
  outl((unsigned int)CLR_PCII,reg->csr);
}


void k2915_crate_enable_lam(int lammask){
  short i = SRE;

  k2915_write16(NAFGEN(30,0,17),&i);
  k2915_write24(NAFGEN(30,13,17),&lammask);
}

void k2915_crate_define_lam(int n){
  short i = SRE;
  int lammask;

  lammask = (0x0001<<(n-1)) & 0x00ffffff;
  k2915_write16(NAFGEN(30,0,17),&i);
  k2915_write24(NAFGEN(30,13,17),&lammask);
}

void k2915_crate_disable_lam(void){
  short i = 0x0000;
  int c = 0x00000000;

  k2915_write16(NAFGEN(30,0,17),&i);
  k2915_write24(NAFGEN(30,13,17),&c);
}

void k2915_get_csrdata(int *ret){
  *ret = inl(reg->csr);
}

void k2915_get_bmcsdata(int *ret){
  *ret = inl(reg->bmcs);
}

void k2915_crate_z(int c){
  short z = 0x0001;

  k2915_write16(CNAFGEN(c,30,0,17),&z);
}

void k2915_crate_c(int c){
  short cl = 0x0002;

  k2915_write16(CNAFGEN(c,30,0,17),&cl);
}

void k2915_crate_seti(int c){
  short ih = 0x0004;

  k2915_write16(CNAFGEN(c,30,0,17),&ih);
}

void k2915_crate_deli(int c){
  short ih = 0x0000;

  k2915_write16(CNAFGEN(c,30,0,17),&ih);
}

int k2915_chkq(void){
  return inl(reg->csr) & NOQ;
}
