/* ccnet.c
 * last modified : 08/11/17 02:13:47 
 *
 * TOYO CC/NET driver
 * Original = pcc.c by Yasu-san
 *
 * Hidetada Baba (RIKEN)
 * baba@ribf.riken.jp
 *
 */

int ccnet_init_register(void);
int ccnet_check_done(int try);
int ccnet_check_emp_fifo(void);
int ccnet_check_full_fifo(void);
int ccnet_check_lam(void);
void ccnet_read_lam(int *data);
int ccnet_control(int cnaf);
int ccnet_read16(int cnaf,short *data);
int ccnet_read24(int cnaf,short *data);
int ccnet_write16(int cnaf,short *data);
int ccnet_write24(int cnaf,int *data);
int ccnet_block_read16(int mode,int cnaf,short *data,int count);
int ccnet_block_read24(int mode,int cnaf,unsigned int *data,int count);
int ccnet_dma_block_read16(int mode,int cnaf,int count);
int ccnet_dma_block_read24(int mode,int cnaf,int count);
void ccnet_crate_reset(void);
void ccnet_rfs_enable_interrupt(void);
void ccnet_rfs_disable_interrupt(void);
void ccnet_pci_enable_interrupt(void);
void ccnet_pci_clear_interrupt(void);
void ccnet_crate_enable_lam(int lammask);
void ccnet_crate_disable_lam(void);
void ccnet_crate_z(int c);
void ccnet_crate_c(int c);
void ccnet_crate_seti(int c);
void ccnet_crate_deli(int c);
void ccnet_get_csrdata(int *ret);
void ccnet_get_bmcsdata(int *ret);
int  ccnet_chkq(void);
void ccnet_getint(void);

void ccnet_clear_fifo(void);
int ccnet_clear_rxfifo(void);
int ccnet_put_data(int cnaf, int val);
int ccnet_get_data(void);

#define CNAFGEN(c,n,a,f) ((((n<<8) | a) << 8) | f)
#define NAFGEN(n,a,f)    ((((n<<8) | a) << 8) | f)

static int lmsk, lastdata;

static int needed_pages;
static char *read_data_buf;
static char *write_data_buf;
static int timeout;

static inline int __get_order(unsigned long size){
  int order;

  size = (size-1) >> (PAGE_SHIFT-1);
  order = -1;
  do {
    size >>= 1;
    order++;
  } while (size);
  return order;
}

int ccnet_init_register(void){
  int i;
  volatile int try;

  try = 0;

  while(try == 0){
    try = 1;

    outl(SYS_RESET|SYS_CAMAC_FRAME_SIZE, pccdev.pccreg.System);
    outl(SYS_READY|SYS_CAMAC_FRAME_SIZE, pccdev.pccreg.System);

    needed_pages = __get_order(MAX_BUFFER_SIZE);

    write_data_buf = (char *)__get_free_pages(GFP_KERNEL, needed_pages);
    outl( virt_to_phys(write_data_buf), pccdev.pccreg.TxAddress );
    if( !write_data_buf ) {
      printk("init_module:cound not get free pages for write...\n");
      return(-ENOMEM);
    }
    read_data_buf = (char *)__get_free_pages(GFP_KERNEL, needed_pages);
    outl(virt_to_phys(read_data_buf), pccdev.pccreg.RxAddress);
    if( !read_data_buf ) {
      printk("init_module:cound not get free pages for read...\n");
      return(-ENOMEM);
    }

    for(i=0;i<200000;i++){
      outb(1,0x80);
    }

    outl(TC_INT_CLR, pccdev.pccreg.TxControl);
    outl(RC_INT_CLR, pccdev.pccreg.RxControl);
    outl(IC_INT_CLR, pccdev.pccreg.IntControl);
    
    for(i=0;i<200000;i++){
      outb(1,0x80);
    }
    
    ccnet_check_lam();
    ccnet_clear_fifo();
    ccnet_put_data(NAFGEN(25, 0, 17), 0);
    if(ccnet_get_data() == -ENODATA) try = 0;
    ccnet_put_data(NAFGEN(25, 0, 24), 0);
    if(ccnet_get_data() == -ENODATA) try = 0;
    if(try == 0){
      for(i=0;i<500000;i++){
	outb(1,0x80);
      }
    }
  }

  return 0;
}

short ccnet_dec(int val, short *q, short *x){
  short sval;

  sval = val & 0xffffff;
  *q = 0x1 & (val >> 24);
  *x = 0x1 & (val >> 25);

  return sval;
}

int ccnet_put_data(int cnaf, int val){
  int cmd, status;

  cmd = 0xE0000000 | cnaf;
  val = val & 0xffffff;

  timeout = 0;
  do {
    status = inl(pccdev.pccreg.TxFifoCount);
    status &= 0xffff;
    timeout++;
  } while ( (status > MAX_FIFO_FRAME-2) && (timeout != PCC_TIMEOUT_PIO ) );
  if( timeout == PCC_TIMEOUT_PIO ) {
    printk("CCNET : put_data:timeout_pio:error...\n");
    return -ETIME;
  }
  outl(val, pccdev.pccreg.TxData1);
  outl(cmd, pccdev.pccreg.TxData2);

  return 0;
}

int ccnet_get_data(void){
  int status, rply, val;

  timeout = 0;
  do {
    status = inl(pccdev.pccreg.RxFifoCount);
    status &= 0xffff;
    timeout++;
  } while ( (status < 2) && (timeout < PCC_TIMEOUT_PIO));
  if( timeout == PCC_TIMEOUT_PIO) {
    printk("CCNET : get_dat:timeout_pio:error...\n");
    return -ENODATA;
  }
  val = inl(pccdev.pccreg.RxData1);
  rply = inl(pccdev.pccreg.RxData2);

  outl(RC_INT_CLR, pccdev.pccreg.RxControl);

  return val;
}

int ccnet_get_int(void){
  int ret = 0, val = 0;

  val = inl(pccdev.pccreg.IntFifoCount);
  if( val >= 2 ) {
    val = inl(pccdev.pccreg.IntData1);
    ret = inl(pccdev.pccreg.IntData2);
  }

  return ret;
}


int ccnet_control(int cnaf){
  int ret;

  ccnet_put_data(cnaf, 0);
  ret = ccnet_get_data();
  if(ret == -ENODATA) printk("Err ccnet_control cnaf(0x%x)\n", cnaf);

  return ret;
}

int ccnet_read16(int cnaf, short *data){
  int ret;

  ccnet_put_data(cnaf, 0);
  ret = ccnet_get_data();
  lastdata = ret;
  *data = ret & 0x0000ffff;

  return ret;
}
int ccnet_read24(int cnaf, short *data){
  int ret;

  ccnet_put_data(cnaf, 0);
  ret = ccnet_get_data();
  lastdata = ret;
  *data = ret & 0xffff;
  *(data + 1) = (ret >> 16) & 0x00ff;

  return ret;
}
int ccnet_write16(int cnaf,short *data){
  int ret;

  ccnet_put_data(cnaf, *data);
  ret = ccnet_get_data();

  return ret;
}
int ccnet_write24(int cnaf,int *data){
  int ret;

  ccnet_put_data(cnaf, *data);
  ret = ccnet_get_data();

  return ret;
}

void ccnet_crate_z(int c){
  ccnet_put_data(NAFGEN(25, 0, 17), 0);
  ccnet_get_data();
}

void ccnet_crate_c(int c){
  ccnet_put_data(NAFGEN(25, 0, 16), 0);
  ccnet_get_data();
}

void ccnet_crate_seti(int c){
  ccnet_put_data(NAFGEN(25, 0, 26), 0);
  ccnet_get_data();
}

void ccnet_crate_deli(int c){
  ccnet_put_data(NAFGEN(25, 0, 24), 0);
  ccnet_get_data();
}

int ccnet_check_done(int try){
  return 0;
}
int ccnet_check_emp_fifo(void){
  return 0;
}

int ccnet_check_full_fifo(void){
  return 0;
}

int ccnet_check_lam(void){
  ccnet_get_int();
  return inl(pccdev.pccreg.IntStatus) & IS_INT;
}

void ccnet_read_lam(int *data){
}


int ccnet_block_read16(int mode,int cnaf,short *data,int count){
  return 0;
}

int ccnet_block_read24(int mode,int cnaf,unsigned int *data,int count){
  return 0;
}

int ccnet_dma_block_read16(int mode,int cnaf,int count){
  return 0;
}

int ccnet_dma_block_read24(int mode,int cnaf,int count){
  return 0;
}

void ccnet_clear_fifo(void){
  int data, status, i;

  // Clear FIFO
  while(1){
    data = inl(pccdev.pccreg.TxControl);
    data |= TC_CLR_FIFO;
    outl(data, pccdev.pccreg.TxControl);
    data = inl(pccdev.pccreg.RxControl);
    data |= RC_CLR_FIFO;
    outl(data, pccdev.pccreg.RxControl);
    data = inl(pccdev.pccreg.IntControl);
    data |= IC_CLR_FIFO;
    outl(data, pccdev.pccreg.IntControl);

    status = inl(pccdev.pccreg.RxFifoCount);
    if(status > 0){
      data = inl(pccdev.pccreg.RxData1);
      status = inl(pccdev.pccreg.RxData2);
      outl(RC_INT_CLR, pccdev.pccreg.RxControl);
      
      for(i=0;i<20000;i++){
	outb(1,0x80);
      }
    }else{
      break;
    }
  }
}

void ccnet_crate_reset(void){
  ccnet_clear_fifo();

  ccnet_crate_z(0);
  ccnet_crate_deli(0);  // Delete Inhibit
  ccnet_check_lam();
}


void ccnet_rfs_enable_interrupt(void){
  ccnet_put_data(NAFGEN(25, 1, 26), 0);
  //ccnet_get_data();

  if(ccnet_get_data() == -ENODATA){
    printk("Err ccnet_enable_int\n");
  }

  pccdev.int_read_counter = 0;
  outl(IC_INT_ENABLE | IC_CLR_FIFO, pccdev.pccreg.IntControl);

  if(ccnet_clear_rxfifo()) printk("Rxfifo enable interrupt\n");
}

void ccnet_rfs_disable_interrupt(void){
  ccnet_put_data(NAFGEN(25, 1, 24), 0);
  //ccnet_get_data();
  if(ccnet_get_data() == -ENODATA){
    printk("Err ccnet_disable_int\n");
  }

  outl(IC_INT_CLR | IC_CLR_FIFO, pccdev.pccreg.IntControl);
  outl(0, pccdev.pccreg.IntControl);

  if(ccnet_clear_rxfifo()) printk("Rxfifo disable interrupt\n");

  pccdev.int_read_counter = 0;
}

void ccnet_pci_enable_interrupt(void){
}

void ccnet_pci_clear_interrupt(void){
}

void ccnet_crate_enable_lam(int lammask){
  lmsk = lammask & 0x00ffffff;
  ccnet_put_data(NAFGEN(25, 1, 16), lmsk);
  //ccnet_get_data();
  if(ccnet_get_data() == -ENODATA){
    printk("Err ccnet_enable_lam\n");
  }

  if(ccnet_clear_rxfifo()) printk("Rxfifo crate enable lam\n");
}

void ccnet_crate_disable_lam(void){
}

void ccnet_get_csrdata(int *ret){
}

void ccnet_get_bmcsdata(int *ret){
}

void ccnet_crate_define_lam(int n){
  lmsk = (1 << (n - 1)) & 0x00ffffff;
  ccnet_put_data(NAFGEN(25, 1, 16), lmsk);
  //ccnet_get_data();
  if(ccnet_get_data() == -ENODATA){
    printk("Err ccnet_define_lam\n");
  }
  if(ccnet_clear_rxfifo()) printk("Rxfifo crate define lam\n");
}

int ccnet_chkq(void){
  int ret = 1;

  if(lastdata & 0x01000000) ret = 0; //NOQ = 1

  return ret;
}

int ccnet_exec_dma(int *cmdbuf, int *rplybuf){
  volatile int try;
  int data, i, length = 0;
  int count, timeout;

  try = 1;

  while(try){
    try = 0;

    count = cmdbuf[1]*8;

    // PCCIOC_KICK_READ
    pccdev.read_flag = 1;
    pccdev.dma_flag = 1;
    outl(count/8, pccdev.pccreg.RxPresetCount);
    data = inl(pccdev.pccreg.RxControl);
    pccdev.int_read_counter = 0;
    pccdev.usr_read_counter = 0;
    
    data = RC_SRT_DMA;
    //data = RC_INT_ENABLE_FORCE_END|RC_INT_ENABLE_PKT_END|RC_SRT_DMA;
    outl(data, pccdev.pccreg.RxControl);
    // End of PCCIOC_KICK_READ
    
    // pcc write
    data = 1;
    timeout = 0;
    while(data){
      if(timeout == PCC_TIMEOUT_PIO){
	printk("CCNET : ccnet_exec:timeout_dma_write:error...\n");
	try = 1;
	//return 0;
      }
      data = inl(pccdev.pccreg.TxFifoCount);
      timeout++;
    }
    
    memcpy(write_data_buf, (char *)(cmdbuf+2), count);
    outl(count/8, pccdev.pccreg.TxPresetCount);
    data = inl(pccdev.pccreg.TxControl);
    pccdev.int_write_counter = 0;
    pccdev.usr_write_counter = 0;
    
    data = TC_SRT_DMA;
    outl(data, pccdev.pccreg.TxControl);
    
    i = 0;
    do {
      i++;
      data=inl(pccdev.pccreg.TxControl);
      if(!(data & TC_SRT_DMA)) {
	data=inl(pccdev.pccreg.TxFifoCount);
	if(data == 0)
	  break;
      }
    } while(!(i==10000000 ) );
    if (i==10000000) {
      printk("CCNET : ccnet_exec:timeout_write(poll):error...\n");
      try = 1;
      //return 0;
    }
    
    pccdev.dma_flag = 0;
    pccdev.int_write_counter = 0;
    length = inl(pccdev.pccreg.TxActualCount);
    // End of ioctl write
    
    
    // pcc read
    pccdev.dma_flag = 1;
    pccdev.read_flag = 0;
    
    
    i = 0;
    do {
      i++;
      data=inl(pccdev.pccreg.RxControl);
      if(!(data& RC_SRT_DMA)) {
	break;
      }
    } while(!(i==10000000 ) );
    if (i==10000000) {
      printk("CCNET : ccnet_exec:timeout_read:error...\n");
      try = 1;
      //return 0;
    }
    
    pccdev.dma_flag = 0;
    pccdev.int_read_counter = 0;
    pccdev.usr_read_counter = 0;
    length = inl(pccdev.pccreg.RxActualCount);
    
    if(try == 0){
      if(count/8 != length){
	printk("count/length %d %d\n", count/8, length);
      }
      memcpy((char *)(rplybuf+1), (char *)&length, sizeof(length));
      memcpy((char *)(rplybuf+2), read_data_buf, length*8);
    }
  }

  return length*8;
}

int ccnet_exec_pio(int *cmdbuf, int *rplybuf){
  volatile int try;
  int i, j, timeout, timeout_flag, cmd_len, total_len;
  int status, actual_len;

  actual_len = 0;
  cmd_len = cmdbuf[1];
  total_len = cmdbuf[1];
  if( (cmd_len > total_len) || (cmd_len <= 0) || (cmd_len > MAX_BUFFER_SIZE/8) || (total_len <=0) )
    return -EINVAL;

  try = 1;

  while(try){
    try = 0;
    
    i = j = 2;
    timeout = 0;
    do {
      timeout_flag = 1;
      if( j < cmd_len*2+2 ) {
	status = inl(pccdev.pccreg.TxFifoCount);
	status &= 0xFFFF;
	if( status < MAX_FIFO_FRAME - 2 ) {
	  outl(cmdbuf[j], pccdev.pccreg.TxData1);
	  outl(cmdbuf[j+1], pccdev.pccreg.TxData2);
	  j += 2;
	  timeout_flag = 0;
	}
      }
      status = inl(pccdev.pccreg.RxFifoCount);
      status &= 0xFFFF;
      if( status >= 2 ) {
	rplybuf[i] = inl(pccdev.pccreg.RxData1);
	rplybuf[i+1] = inl(pccdev.pccreg.RxData2);
	i += 2;
	timeout_flag = 0;
      }
      if(timeout_flag==0)
	timeout = 0;
      else
	timeout++;
    } while ( (i < total_len*2+2) && (timeout != PCC_TIMEOUT_PIO*100 ) );
    if( timeout ==  PCC_TIMEOUT_PIO*100) {
      try = 1;
      //return -ETIME;
    }
    actual_len = (i-2)/2;
    rplybuf[1] = actual_len;
    outl(RC_INT_CLR, pccdev.pccreg.RxControl);
  }
  return actual_len*8+8;
}

int ccnet_exec(int *cmdbuf, int *rplybuf){
  int status;

  if(cmdbuf[1] > NUM_FRAME_SWITCH){
    status = ccnet_exec_dma(cmdbuf, rplybuf);
  }else{
    status = ccnet_exec_pio(cmdbuf, rplybuf);
  }

  return status;
}

int ccnet_clear_rxfifo(void){
  int cnt, x, y;
  volatile int i;

  cnt = inl(pccdev.pccreg.RxFifoCount);
  for(i=0;i<cnt;i++){
    x = inl(pccdev.pccreg.RxData1);
    y = inl(pccdev.pccreg.RxData2);
    printk("rxfifo %d data 0x%08x / rply 0x%08x\n", i, x, y);
  }
  outl(RC_INT_CLR, pccdev.pccreg.RxControl);

  return cnt;
}
