/*
 * TOYO CC/PCI PCI Interface to CC/7700 Controller
 * 
 * Hidetada Baba
 * baba@ribf.riken.jp
 * last modified : 07/03/01 15:27:24 
 */

#include "cc7700.h"

#if LINUX_VERSION_CODE >= 0x020600
void cc77_init_register(unsigned int irq,resource_size_t ior,resource_size_t pbor);
#else
void cc77_init_register(unsigned int irq,unsigned long ior,unsigned long pbor);
#endif

int cc77_check_done(int try);
int cc77_check_emp_fifo(void);
int cc77_check_full_fifo(void);
int cc77_check_lam(void);
void cc77_read_lam(int *data);
int cc77_control(int cnaf);
int cc77_read16(int cnaf,short *data);
int cc77_read24(int cnaf,short *data);
int cc77_write16(int cnaf,short *data);
int cc77_write24(int cnaf,int *data);
int cc77_block_read16(int mode,int cnaf,short *data,int count);
int cc77_block_read24(int mode,int cnaf,unsigned int *data,int count);
int cc77_dma_block_read16(int mode,int cnaf,int count);
int cc77_dma_block_read24(int mode,int cnaf,int count);
void cc77_crate_reset(void);
void cc77_rfs_enable_interrupt(void);
void cc77_rfs_disable_interrupt(void);
void cc77_pci_enable_interrupt(void);
void cc77_pci_clear_interrupt(void);
void cc77_crate_enable_lam(int lammask);
void cc77_crate_disable_lam(void);
void cc77_crate_z(int c);
void cc77_crate_c(int c);
void cc77_crate_seti(int c);
void cc77_crate_deli(int c);
void cc77_get_csrdata(int *ret);
void cc77_get_bmcsdata(int *ret);

static int done_try,lmsk;

#if LINUX_VERSION_CODE >= 0x020600
struct pci_struct{
  resource_size_t base1;
  resource_size_t base2;
  unsigned int irq;
};
#else
struct pci_struct{
  unsigned long base1;
  unsigned long base2;
  unsigned int irq;
};
#endif

struct pci_struct pci_table[] ={
  {0, 0, 0}
};

#if LINUX_VERSION_CODE >= 0x020600
struct pci_reg{
  resource_size_t csr;
  resource_size_t cnaf;
  resource_size_t tcr;
  resource_size_t srr;
  resource_size_t fifo;
  resource_size_t mwa;
  resource_size_t mwtc;
  resource_size_t mra;
  resource_size_t mrtc;
  resource_size_t ics;
  resource_size_t bmcs;  
};
#else
struct pci_reg{
  unsigned long csr;
  unsigned long cnaf;
  unsigned long tcr;
  unsigned long srr;
  unsigned long fifo;
  unsigned long mwa;
  unsigned long mwtc;
  unsigned long mra;
  unsigned long mrtc;
  unsigned long ics;
  unsigned long bmcs;  
};
#endif

struct pci_reg reg[] ={
  {CSR, CNAF, TCR, SRR, FIFO, MWA, MWTC, MRA, MRTC, ICS, BMCS,},
};

#if LINUX_VERSION_CODE >= 0x020600
void cc77_init_register(unsigned int irq,resource_size_t ior,resource_size_t pbor){
#else
void cc77_init_register(unsigned int irq,unsigned long ior,unsigned long pbor){
#endif

  pci_table->irq   = irq;
  pci_table->base1 = ior;
  pci_table->base2 = pbor;
  
  reg->csr  += pbor; /* for Create Control */
  reg->cnaf += pbor; /* for CNAF */
  reg->tcr  += pbor; /* Not use */
  reg->srr  += pbor; /* for GO request */

  reg->fifo += pbor;  /* for Data Transfer */
  reg->mwa  += ior;  /* Not use */
  reg->mwtc += ior;  /* Not use */
  reg->mra  += ior;  /* Not use */
  reg->mrtc += ior;  /* Not use */
  reg->ics  += ior;  /* Not use */
  reg->bmcs += ior;  /* Not use */

}

int cc77_check_done_retq(int try){
  int val;

  done_try = 0;
  while (!((val = inl(reg->csr)) & DONE)){
    done_try++;
    if(done_try > try) {
      /* printk("check_done: Timeout, Waiting for DONE\n"); */
      return -1;
      break;
    }
  }

  return val & NOQ;
}

int cc77_check_done(int try){
  int ret = 0;

  done_try = 0;
  while (!(inl(reg->csr) & DONE)){
    done_try++;
    if(done_try > try) {
      /* printk("check_done: Timeout, Waiting for DONE\n"); */
      ret = 1;
      break;
    }
  }
  return ret;
}

int cc77_check_emp_fifo(void){
  int ret = 0;

  return ret;
}

int cc77_check_full_fifo(void){
  int ret = 0;

  return ret;
}

int cc77_check_lam(void){
  return (inl(reg->cnaf) & lmsk) & 0x00ffffff;
}

void cc77_read_lam(int *data){
  *data = inl(reg->cnaf) & 0x00ffffff;
}

int cc77_control(int cnaf){
  outl(cnaf,reg->cnaf);
  outl((unsigned int)GO,reg->srr);
  return cc77_check_done_retq(MAX_DONE_TRY);
}

int cc77_read16(int cnaf,short *data){
  int ret;
  outl(cnaf,reg->cnaf);
  outl((unsigned int)GO,reg->srr);
  ret = cc77_check_done_retq(MAX_DONE_TRY);
  *data = inl(reg->fifo);
  return ret;
}

int cc77_read24(int cnaf,short *data){
  unsigned int tdata;
  int ret;
  outl(cnaf,reg->cnaf);
  outl((unsigned int)GO,reg->srr);
  ret = cc77_check_done_retq(MAX_DONE_TRY);
  tdata = inl(reg->fifo);
  *data = tdata & 0xffff;
  *(data+1) = (tdata>>16) & 0x00ff;
  return ret;
}

int cc77_write16(int cnaf,short *data){
  outl(cnaf,reg->cnaf);
  outl((unsigned int)*data,reg->fifo);
  outl((unsigned int)GO,reg->srr);
  return cc77_check_done_retq(MAX_DONE_TRY);
}

int cc77_write24(int cnaf,int *data){
  outl(cnaf,reg->cnaf);
  outl((unsigned int)*data,reg->fifo);
  outl((unsigned int)(SINGLE24 | GO),reg->csr);
  return cc77_check_done_retq(MAX_DONE_TRY);
}

int cc77_block_read16(int mode,int cnaf,short *data,int count){
  int ret = 0,btry = 0, br = 0;
  unsigned int tdata;

  outl(cnaf,reg->cnaf);

  while(1){
    outl((unsigned int)GO,reg->srr);
    if(mode == QREPEAT){
      if(cc77_check_done_retq(MAX_DONE_TRY) != 0){
	br = 1;
      }
    }else if(mode == QSTOP){
      if(cc77_check_done_retq(MAX_DONE_TRY) == 0){
	br = 1;
      }
    }else{
      if(cc77_check_done(MAX_DONE_TRY) == 1){
	/* printk("Error DONE in block_read16.\n"); */
	break;
      }
    }
    if(br == 0){
      tdata = inl(reg->fifo);
      *(data+btry) = tdata & 0xffff;
      btry++;
    }
    if(btry == count || br == 1){
      /* printk("btry = count\n"); */
      break;
    }
  }

  ret = btry;

  return ret;
}

/*
  int block_read24(int mode,int cnaf,short *data,int count){
  int ret = 0,btry = 0;
  unsigned int tdata;

  outl(cnaf,reg->cnaf);

  while(1){
  outl((unsigned int)GO,reg->srr);
  if(check_done(MAX_DONE_TRY) == 1){
  break;
  }
  tdata = inl(reg->fifo);
  *(data+btry) = tdata & 0xffff;
  btry++;
  *(data+btry) = (tdata >>16) & 0x00ff;
  btry++;
  if(btry == count*2){
  break;
  }
  }

  ret = btry;

  return ret;
  }
*/

int cc77_block_read24(int mode,int cnaf,unsigned int *data,int count){
  int ret = 0,btry = 0;

  outl(cnaf,reg->cnaf);

  while(1){
    outl((unsigned int)GO,reg->srr);
    if(cc77_check_done(MAX_DONE_TRY) == 1){
      /* printk("Error DONE in block_read24.\n"); */
      break;
    }
    *(data+btry) = inl(reg->fifo) & 0x00ffffff;
    btry++;
    if(btry == count){
      /* printk("btry = count\n"); */
      break;
    }
  }

  ret = btry;

  return ret;
}


int cc77_dma_block_read16(int mode,int cnaf,int count){
  int ret = 0;

  /* Not use */

  return ret;
}

int cc77_dma_block_read24(int mode,int cnaf,int count){
  int ret = 0;

  /* Not use */

  return ret;
}

int cc77_dma_block_write16(int mode,int cnaf,int count){
  int ret = 0;

  /* Not use */

  return ret;
}

void cc77_crate_reset(void){
  cc77_crate_z(0);
}

void cc77_rfs_enable_interrupt(void){
  outl(CC_EI,reg->csr);
}

void cc77_rfs_disable_interrupt(void){
  outl(0x00000000,reg->csr);
}

void cc77_pci_enable_interrupt(void){
  /* Not use */
}

void cc77_pci_clear_interrupt(void){
  /* Not use */
}

void cc77_crate_enable_lam(int lammask){
  lmsk = lammask & 0x00ffffff;
  //outl(CC_EI,reg->csr);
}
void cc77_crate_define_lam(int n){
  lmsk = (0x0001<<(n-1)) & 0x00ffffff;
}

void cc77_crate_disable_lam(void){
  outl(0x00000000,reg->csr);
}

void cc77_get_csrdata(int *ret){
  /* Not use */
}

void cc77_get_bmcsdata(int *ret){
  /* Not use */
}

void cc77_crate_z(int c){
  outl(CC_Z,reg->csr);
}

void cc77_crate_c(int c){
  outl(CC_C,reg->csr);
}

void cc77_crate_seti(int c){
  outl(CC_I,reg->csr);
}

void cc77_crate_deli(int c){
  outl(0x00000000,reg->csr);
}

int cc77_chkq(void){
  return inl(reg->csr) & NOQ;
}
