/* last modified : 13/12/13 16:13:10 
 *
 * gto/cmdgscrgto/cmdgscrgto.c
 *
 * Command line program for Gated Scaler GTO
 *
 * H.B. 2013
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>

#include "bbescape.h"
#include "bbtime.h"
#include "libgscrgto.h"

enum emode {READ, VERSION, SETTING};
enum econt {SINGLE, CONTINUOUS};
enum eblock {NORMAL, BLOCK, NOTHING};
enum estore {NONE, STORE, FREAD};
enum eset  {GATEENA, SOFTVETO, TESTLED, LEVEL, PULSE, INIT, CLEAR, ID};
double vreflesh = 2.0; // reflesh rate for continuous readout
enum emode mode = READ;       // mode readout/version/setting
enum econt cont = SINGLE;     // for continuous readout
enum eset  set = CLEAR;
enum eblock blk = NORMAL; 
enum estore store = NONE;
int of = 0;
char ofstr[2][4] = {"OFF\0", "ON\0\0"};
char id[2] = "  ";
int sz = 0;

char data[GSCRGTO_SIZEOF_DATA];
char fname[256] = {0}, fdir[256] = {"./\0"};
FILE *fd = NULL;
struct stgscrdata gscr, igscr, pgscr;
static int first = 1;

void show_data(void){
  int i;
  double time1k, rate;

  if(first){
    memcpy((char *)&igscr, (char *)&gscr, sizeof(gscr));
  }else{
    if(igscr.cnt1k > gscr.cnt1k){
      memcpy((char *)&igscr, (char *)&gscr, sizeof(gscr));
      first = 1;
    }
  }

  if(gscr.gateena){
    printf("Gate Enable  / ");
  }else{
    printf("Gate Disable / ");
  }
  if(gscr.softveto){
    printf("Soft Veto On\n");
  }else{
    printf("Soft Veto Off\n");
  }
  if(gscr.veto){
    printf("Veto Status On   / ");
  }else{
    printf("Veto Status Off  / ");
  }
  if(gscr.level){
    printf("Level Output On\n");
  }else{
    printf("Level Output Off\n");
  }

  printf("1kHz = %13u  / gated 1kHz = %13u\n", gscr.cnt1k, gscr.gcnt1k);
  printf("Gate Number = %u\n", gscr.gatenum);

  printf("           Low value  /    Current rate /    Total rate \n");
  for(i=0;i<20;i++){
    printf("Scr[%2d] %13u ", i, gscr.scr[i]);
    if(!first){
      time1k = gscr.gcnt1k - pgscr.gcnt1k;
      rate = gscr.scr[i] - pgscr.scr[i];
      if(time1k != 0.){
	rate = rate/time1k;
      }else{
	rate = 0.;
      }
      if(rate < 1.){
	printf(" / %8.3f (Hz) ", rate*1000.);
      }else if(rate > 1000.){
	printf(" / %8.3f (MHz)", rate/1000.);
      }else{
	printf(" / %8.3f (kHz)", rate);
      }


      time1k = gscr.gcnt1k - igscr.gcnt1k;
      rate = gscr.scr[i] - igscr.scr[i];
      if(time1k != 0.){
	rate = rate/time1k;
      }else{
	rate = 0.;
      }
      if(rate < 1.){
	printf(" / %8.3f (Hz) ", rate*1000.);
      }else if(rate > 1000.){
	printf(" / %8.3f (MHz)", rate/1000.);
      }else{
	printf(" / %8.3f (kHz)", rate);
      }

    }
    printf("\n");
  }

  first = 0;
  memcpy((char *)&pgscr, (char *)&gscr, sizeof(gscr));
}

void quit(void){
  gscr_close();
  if(fd) fclose(fd);
  exit(0);
}

int chkof(char *of){
  if(!strncmp(of, "ON", 2)){
    return 1;
  }else if(!strncmp(of, "OFF", 3)){
    return 0;
  }
  return -1;
}

void help(void){
  printf("cmdgscrgto [OPTIONS] hostname\n");
  printf(" -h        : this help\n");
  printf(" -n        : get version information\n");
  printf(" -f XX     : set identifier\n");
  printf(" -s        : continuous readout\n");
  printf(" -S        : continuous readout and store data\n");
  printf(" -R FILE   : readout from gscr file\n");
  printf(" -d DIR    : directory for storing data\n");
  printf(" -b        : block other connection during continuous readout\n");
  printf(" -r RATE   : reflesh rate for continuous readout\n");
  printf(" -g ON/OFF : Gate Enable On or Off\n");
  printf(" -v ON/OFF : Soft Veto On or Off\n");
  printf(" -t ON/OFF : Test LED On or Off\n");
  printf(" -l ON/OFF : Level output On or Off\n");
  printf(" -p        : Pulse output\n");
  printf(" -c        : Clear data\n");
  printf(" -i        : Initialize\n");
  printf("\n\n");
  printf("EXAMPLE : \n");
  printf(" cmdgscrgto -s -r 0.5     : continuous readout 0.5s each\n");
  printf(" cmdgscrgto -s -b         : continuous readout 1s each, block other connection\n");
}

int chkopt(int argc, char *argv[]){
  int val;
  while((val = getopt(argc, argv, "hnsbcpibr:R:S:g:t:l:v:f:d:")) != -1){
    switch(val){
    case 's':
      cont = CONTINUOUS;
      break;
    case 'S':
      cont = CONTINUOUS;
      store = STORE;
      break;
    case 'R':
      cont = CONTINUOUS;
      store = FREAD;
      blk = NOTHING;
      strncpy(fname, optarg, sizeof(fname));
      break;
    case 'd':
      strncpy(fdir, optarg, sizeof(fdir));
      break;
    case 'b':
      blk = BLOCK;
      break;
    case 'r':
      vreflesh = strtod(optarg, NULL);
      break;
    case 'h':
      help();
      quit();
      break;
    case 'n':
      mode = VERSION;
      break;
    case 'g':
      mode = SETTING;
      set = GATEENA;
      if((of = chkof(optarg)) < 0){
	printf("Invalid ON/OFF %s\n", optarg);
	quit();
      }
      break;
    case 'v':
      mode = SETTING;
      set = SOFTVETO;
      if((of = chkof(optarg)) < 0){
	printf("Invalid ON/OFF %s\n", optarg);
	quit();
      }
      break;
    case 't':
      mode = SETTING;
      set = TESTLED;
      if((of = chkof(optarg)) < 0){
	printf("Invalid ON/OFF %s\n", optarg);
	quit();
      }
      break;
    case 'l':
      mode = SETTING;
      set = LEVEL;
      if((of = chkof(optarg)) < 0){
	printf("Invalid ON/OFF %s\n", optarg);
	quit();
      }
      break;
    case 'p':
      mode = SETTING;
      set = PULSE;
      break;
    case 'c':
      mode = SETTING;
      set = CLEAR;
      break;
    case 'i':
      mode = SETTING;
      set = INIT;
      break;
    case 'f':
      mode = SETTING;
      set = ID;
      if(strlen(optarg) < 2){
	printf("ID is more than 2 characters\n");
	quit();
      }
      memcpy(id, optarg, 2);
      break;
    default:
      printf("Invalid option!! %c\n", val);
      return 0;
      break;
    }
  }
  return 1;
}

void dsleep(double ref){
  usleep((int)(ref*1000*1000));
}

int main(int argc, char *argv[]){
  char ver[12] = {0};

  signal(SIGINT, (void *)quit);

  if(argc < 2){
    help();
    quit();
  }

  if(!chkopt(argc, argv)){
    quit();
  }

  if(store != FREAD){
    if(!gscr_open(argv[argc-1])){
      printf("Cannot connect %s\n", argv[argc-1]);
      quit();
    }
  }


  switch(mode){
  case READ:
    /* next */
    break;
  case VERSION:
    gscr_version(ver);
    printf("Version = %s\n", ver);
    quit();
    break;
  case SETTING:
    if(set == GATEENA){
      gscr_gateena(of);
      printf("Gate Enable = %s\n", ofstr[of]);
    }else if(set == SOFTVETO){
      gscr_softveto(of);
      printf("Soft Veto = %s\n", ofstr[of]);
    }else if(set == TESTLED){
      gscr_testled(of);
      printf("Test LED = %s\n", ofstr[of]);
    }else if(set == LEVEL){
      gscr_level(of);
      printf("Level output = %s\n", ofstr[of]);
    }else if(set == PULSE){
      gscr_pulse();
      printf("Pulse output\n");
    }else if(set == CLEAR){
      gscr_clear();
      printf("Scaler cleared\n");
    }else if(set == INIT){
      gscr_init();
      printf("Initialized\n");
    }else if(set == ID){
      gscr_id(id);
      printf("Set ID %c%c\n", id[0], id[1]);
      gscr_version(ver);
      printf("Version = %s\n", ver);
    }
    quit();
  default:
    break;
  }

  if(cont == CONTINUOUS && store == STORE){
    sprintf(fname, "%s%s.gscr", fdir, timeforfilename(0));
    printf("Output file = %s\n", fname);
    if((fd = fopen(fname, "w")) == NULL){
      printf("Cannot open write file %s\n", fname);
      quit();
    }
  }

  if(store == FREAD){
    if((fd = fopen(fname, "r")) == NULL){
      printf("Cannot open read file %s\n", fname);
      quit();
    }else{
      fread(data, 1, GSCRGTO_SIZEOF_DATA, fd);
    }
  }
    

  while(1){
    if(cont == CONTINUOUS){
      if(blk == NORMAL){
	gscr_open(argv[argc-1]);
      }
      if(store == FREAD){
	sz = fread(data, 1, GSCRGTO_SIZEOF_DATA, fd);
	if(sz != GSCRGTO_SIZEOF_DATA){
	  quit();
	}
      }else{	
	gscr_data(data);
      }
      esc_clear();
      esc_curpos(0, 0);
      printf("Reflesh = %2.1fs", vreflesh);
      if(store == STORE){
	printf("  (data in %s)", fname);
      }else if(store == FREAD){
	printf("  (data from %s)\n", fname);
      }
      printf("\n");
    }
    gscr_conv(data, &gscr);
    show_data();

    if(fd && store == STORE){
      fwrite(data, 1, GSCRGTO_SIZEOF_DATA, fd);
    }
    if(cont == SINGLE) quit();
    if(cont == CONTINUOUS && blk == NORMAL){
      gscr_close();
    }
    if(store != FREAD) dsleep(vreflesh);
  }
  quit();

  return 0;
}
